﻿using System;
using System.Threading;
using Microsoft.SPOT;
using Microsoft.SPOT.Hardware;
using SecretLabs.NETMF.Hardware;
using SecretLabs.NETMF.Hardware.NetduinoPlus;
using IRSky;

/// +------------------------------------------------------------------------------------------------------------------------------+
/// ¦                                                   TERMS OF USE: MIT License                                                  ¦
/// +------------------------------------------------------------------------------------------------------------------------------¦
/// ¦Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    ¦
/// ¦files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    ¦
/// ¦modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software¦
/// ¦is furnished to do so, subject to the following conditions:                                                                   ¦
/// ¦                                                                                                                              ¦
/// ¦The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.¦
/// ¦                                                                                                                              ¦
/// ¦THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          ¦
/// ¦WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         ¦
/// ¦COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   ¦
/// ¦ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         ¦
/// +------------------------------------------------------------------------------------------------------------------------------+

namespace IRSky
{
    public class Program
    {
        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Program Main Loop. It all starts here
        /// </summary>
        /// <history>
        ///   20 Jan 14  Cynic - originally written
        /// </history>
        public static void Main()
        {
            // call the test routine
            TestSkyPlusIRDecoder();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Tests the SKY+ IR decoder routines. Never Leaves here. Expects a specific 
        /// hardware setup as discussed below.
        /// 
        /// Hardware:
        ///   Netduino Plus 2
        ///   Sparkfun IR Receiver Breakout SEN-08554 configured on pin Pins.GPIO_PIN_D7
        ///   Sky+ IR Remote with TV mode configured for a Phillips TV
        /// Optional Hardware:
        ///   Sparkfun 7-Segment Serial Display COM-11441 configured for SPI on GPIO_PIN_D9
        ///   Sparkfun 7-Segment Serial Display COM-11441 configured for SPI on GPIO_PIN_D10
        /// 
        /// </summary>
        /// <history>
        ///   20 Jan 14  Cynic - originally written
        /// </history>
        public static void TestSkyPlusIRDecoder()
        {
            // this contains the information we write to the 7 Segment displays
            byte[] outBuffer = new byte[4];

            // set up the onboard button. For diagnostic reasons it's nice to be able to 
            // press this and hit a debug break so we can look at the state of things
            // it is entirely optional if this is configured
            InputPort button = new InputPort(Pins.ONBOARD_SW1, false, Port.ResistorMode.Disabled);

            // now we set up the 7-Segment Serial Displays - these are not strictly necessary
            // it is just a nice way of looking at the IR remote codes. It also could be done
            // by sending them out a serial port & etc.
 
            // Set up a 7-Segment Serial Display on Pin 10
            SPI.Configuration Device1 = new SPI.Configuration(
                Pins.GPIO_PIN_D10, // SS-pin
                false,             // SS-pin active state
                0,                 // The setup time for the SS port
                0,                 // The hold time for the SS port
                false,             // The idle state of the clock -->> NOTE: this MUST be FALSE to work with SparkFun 7 segment displays 
                false,             // The sampling clock edge
                10,                // The SPI clock rate in KHz
                SPI_Devices.SPI1   // The used SPI bus (refers to a MOSI MISO and SCLK pinset)
            );

            // Set up a 7-Segment Serial Display on Pin 9
            SPI.Configuration Device2 = new SPI.Configuration(
                Pins.GPIO_PIN_D9,  // SS-pin
                false,             // SS-pin active state
                0,                 // The setup time for the SS port
                0,                 // The hold time for the SS port
                false,             // The idle state of the clock -->> NOTE: this MUST be FALSE to work with SparkFun 7 segment displays 
                false,             // The sampling clock edge
                10,                // The SPI clock rate in KHz
                SPI_Devices.SPI1   // The used SPI bus (refers to a MOSI MISO and SCLK pinset)
            );

            // Set up SPI bus and point it at 7 Segment Display 1
            SPI SPIBus = new SPI(Device1);
            // reset 7 Segment Display 1
            FillDisplayBufferWithBottomFourBytesOfInt32(0x78787878, outBuffer);
            SPIBus.Write(outBuffer);
            // reset 7 Segment Display 2
            SPIBus.Config = Device2;
            SPIBus.Write(outBuffer);

            // create a new decoder app class, the Sparkfun IR Receiver Breakout is on the pin
            // in the constructor call
            RC6RC5Decoder rc6rc5Decoder = new RC6RC5Decoder(Pins.GPIO_PIN_D7);

            while (true)
            {
                // we do not want to be continuously processable
                Thread.Sleep(10);

                // This is diagnostic code - we read the buttons state and if pressed
                // we execute the if statement below
	            bool buttonState = button.Read();
                if (buttonState == true)
                {
                    // set a break point below here. The code will stop when
                    // you press the button and you can inspect things.
                    int testVal = rc6rc5Decoder.ResetCount;
                    testVal = rc6rc5Decoder.ProcessedIRCodeCount;
                    testVal = rc6rc5Decoder.ConsumedIRCodeCount;
                    testVal = rc6rc5Decoder.LastResetReason;
                }

                // display some diagnostic information on Device 1. I like to output
                // the ResetCount. This is the total number of signal packets including
                // the discarded ones. It is amazing how much garbage signal the decoder
                // has to chuck out in between the real signals. Mostly it is fluorescent
                // lights kicking out all sorts of stray IR
                SPIBus.Config = Device1;
                int diagnosticValue = rc6rc5Decoder.ResetCount;
                //int diagnosticValue = rc6rc5Decoder.ProcessedIRCodeCount;
                FillDisplayBufferWithBottomFourNibblesOfInt32(diagnosticValue, outBuffer);
                SPIBus.Write(outBuffer);

                // have we got any data?
                if (rc6rc5Decoder.IsDataAvailable() == true)
                {
                    // yes, we have, set up to display on device 2
                    SPIBus.Config = Device2;
                    // get the data
                    int irCode = rc6rc5Decoder.GetData();
                    // prepare it for display 
                    FillDisplayBufferWithBottomFourNibblesOfInt32(irCode, outBuffer);
                    // output to the display 2
                    SPIBus.Write(outBuffer);
                }
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Utility function to populate a bute array with the contents of the bottom
        /// four nibbles of an int32.
        /// 
        /// This will cause the bottom 4 nibbles to be placed in output buffer as hex values
        /// </summary>
        /// <param name="intToDisplay">the int we display</param>
        /// <param name="writeBuffer">four slot byte[] array to fill</param>
        /// <history>
        ///   20 Jan 14  Cynic - originally written
        /// </history>
        public static void FillDisplayBufferWithBottomFourNibblesOfInt32(int intToDisplay, byte[] outBuffer)
        {
            outBuffer[0] = (byte)((0xf000 & intToDisplay) >> 12);
            outBuffer[1] = (byte)((0x0f00 & intToDisplay) >> 8);
            outBuffer[2] = (byte)((0x00f0 & intToDisplay) >> 4);
            outBuffer[3] = (byte)((0x000f & intToDisplay) >> 0);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Utility function to populate a bute array with the contents of the four bytes
        /// of an int32.
        /// 
        /// This will cause the 4 bytes to be placed in output buffer as hex values
        /// </summary>
        /// <param name="intToDisplay">the int we display</param>
        /// <param name="writeBuffer">four slot byte[] array to fill</param>
        /// <history>
        ///   20 Jan 14  Cynic - originally written
        /// </history>
        public static void FillDisplayBufferWithBottomFourBytesOfInt32(int intToDisplay, byte[] outBuffer)
        {
            outBuffer[0] = (byte)((0xff000000 & intToDisplay) >> 24);
            outBuffer[1] = (byte)((0x00ff0000 & intToDisplay) >> 16);
            outBuffer[2] = (byte)((0x0000ff00 & intToDisplay) >> 8);
            outBuffer[3] = (byte)((0x000000ff & intToDisplay) >> 0);
        }

    }
}
