using System;
using System.Collections.Generic;
using System.Text;

/// +------------------------------------------------------------------------------------------------------------------------------+
///                                                    TERMS OF USE: MIT License                                                  
/// +------------------------------------------------------------------------------------------------------------------------------
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    
/// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
/// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
/// is furnished to do so, subject to the following conditions:                                                                   
///                                                                                                                               
/// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
///                                                                                                                               
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
/// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
/// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
/// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
/// +------------------------------------------------------------------------------------------------------------------------------+

namespace PropScriptBitInspector
{
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// <summary>
    /// A class to transform binary data into strings and vice versa
    /// </summary>
    /// <history>
    ///    10 Apr 09  Cynic - Started
    /// </history>
    class BinaryDataConverter
    {
        public const int DEFAULT_PROGRAM_COUNTER_DIGITS = 4;
        public const int DEFAULT_HEX_DIGITS = 2;
        public const int DEFAULT_BINARY_DIGITS = 8;
        public const int NUMBYTES_IN_PASM_INIT_HEADER = 24;
        public const int NUMBYTES_IN_PASM_INIT_PRIMARYHEADER = 16;
        private const int NUM_BITS_IN_OPCODE_INSTR = 6;
        private const int NUM_BITS_IN_OPCODE_ZCRI = 4;
        private const int NUM_BITS_IN_OPCODE_CON = 4;
        private const int NUM_BITS_IN_OPCODE_DEST = 9;
        private const int NUM_BITS_IN_OPCODE_SRC = 9;

        public const string LABEL_ADDRESS_MARKER = "@";
        public const char LABEL_ADDRESS_MARKER_AS_CHAR = '@';
        public const string LABEL_COLUMN_MARKER = "^^^^";
        public const string LABEL_COLUMN_MARKER_BLANK = "    ";

        public const int MAX_BYTES_IN_COGMEMORY = 2048;
        public const int MAX_LONGS_IN_COGMEMORY = (MAX_BYTES_IN_COGMEMORY / 4);


        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts 4 bytes to a spaced hex string. The four bytes are assumed to be
        /// passed in in little endian format
        /// </summary>
        /// <param name="byte0">The lowest byte to convert</param>
        /// <param name="byte1">The next byte to convert</param>
        /// <param name="byte2">The next byte to convert</param>
        /// <param name="byte3">The highest byte to convert</param>
        /// <param name="interpretEverythingAsPASM">if true we do not attempt to find the
        /// spin parts of the code - everything gets interpreted as if it were PASM</param>
        /// <param name="wantProgramCounter">if true we want a program counter column</param>
        /// <param name="wantPasmCodes">if true we want PASM codes</param>
        /// <param name="wantProgCounter">if true we want the program counter line</param>
        /// <param name="programCounter">the value of the program counter for this line</param>
        /// <param name="wantBinaryValues">if true we want the binary values column</param>
        /// <param name="removeLittleEndian">if true we remove the litte endianness</param>
        /// <param name="programStart">The offset to the program start</param>
        /// <param name="stackBase">The offset from the stack base</param>
        /// <param name="stackStart">The offset from the stack start</param>
        /// <param name="variableBase">The offset to the variable base</param>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        public static string GetBinaryFileDisplayLine(bool interpretEverythingAsPASM, bool removeLittleEndian, byte byte0, byte byte1, byte byte2, byte byte3, bool wantProgCounter, int programCounter, bool wantBinaryValues, bool wantPasmCodes, int programStart, int variableBase, int stackBase, int stackStart)
        {
            string tmpStr;
            StringBuilder sb = new StringBuilder();
            StringBuilder sbBinary = new StringBuilder();

            byte displayByte0;
            byte displayByte1;
            byte displayByte2;
            byte displayByte3;

            string binaryStr;
            string strINSTR;
            string strZCRI;
            string strCON;
            string strDEST;
            string strSRC;

            if (removeLittleEndian == true)
            {
                displayByte0 = byte3;
                displayByte1 = byte2;
                displayByte2 = byte1;
                displayByte3 = byte0;
            }
            else
            {
                displayByte0 = byte0;
                displayByte1 = byte1;
                displayByte2 = byte2;
                displayByte3 = byte3;
            }

            // do we want the program counter column?
            if (wantProgCounter == true)
            {
                // yes, we do
                tmpStr = System.Convert.ToString(programCounter, 16);
                for (int i = 0; i < DEFAULT_PROGRAM_COUNTER_DIGITS; i++)
                {
                    // add on leading zeros till we pad it out
                    if (tmpStr.Length >= DEFAULT_PROGRAM_COUNTER_DIGITS) break;
                    tmpStr = "0" + tmpStr;
                }
                sb.Append(tmpStr.ToUpper()+":");
                sb.Append(" ");          
            }

            sb.Append(" ");
            // convert the byte values to hex
            tmpStr = System.Convert.ToString(displayByte0, 16);
            sb.Append(PadWithLeadingZeros(tmpStr.ToUpper(),DEFAULT_HEX_DIGITS));
            sb.Append(" ");
            tmpStr = System.Convert.ToString(displayByte1, 16);
            sb.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_HEX_DIGITS));
            sb.Append(" ");
            tmpStr = System.Convert.ToString(displayByte2, 16);
            sb.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_HEX_DIGITS));
            sb.Append(" ");
            tmpStr = System.Convert.ToString(displayByte3, 16);
            sb.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_HEX_DIGITS));
            sb.Append(" ");

            // convert the byte values to binary
            sbBinary.Append(" ");
            tmpStr = System.Convert.ToString(displayByte0, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
            sbBinary.Append(" ");
            tmpStr = System.Convert.ToString(displayByte1, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
            sbBinary.Append(" ");
            tmpStr = System.Convert.ToString(displayByte2, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
            sbBinary.Append(" ");
            tmpStr = System.Convert.ToString(displayByte3, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
            sbBinary.Append(" ");
            // do we actually want to use these in the output?
            if (wantBinaryValues == true)
            {
                // yes we do, tack them on
                sb.Append(sbBinary);
            }

            // do we want to put out the Prop assembler codes?
            if (wantPasmCodes == true)
            {
                // are we in the header section?
                if (programCounter < NUMBYTES_IN_PASM_INIT_HEADER)
                {
                    if(programCounter < NUMBYTES_IN_PASM_INIT_PRIMARYHEADER)
                    {
                        // yes we are
                        sb.Append("  ");
                        sb.Append(LABEL_COLUMN_MARKER);
                        sb.Append("  ");
                        sb.Append("Header (Primary)");
                        sb.Append("  ");
                    }
                    else
                    {
                        // yes we are
                        sb.Append("  ");
                        sb.Append(LABEL_COLUMN_MARKER);
                        sb.Append("  ");
                        sb.Append("Header (Secondary)");
                        sb.Append("  ");
                    }
                }
                else
                {
                    // convert the byte values to binary, make sure it is in little endian order
                    sbBinary = new StringBuilder();
                    tmpStr = System.Convert.ToString(byte3, 2);
                    sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
                    tmpStr = System.Convert.ToString(byte2, 2);
                    sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
                    tmpStr = System.Convert.ToString(byte1, 2);
                    sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
                    tmpStr = System.Convert.ToString(byte0, 2);
                    sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));

                    // get the various components as binary strings
                    binaryStr = sbBinary.Replace(" ", "").ToString();
                    strINSTR = binaryStr.Substring(0, 6);
                    strZCRI = binaryStr.Substring(6, 4);
                    strCON = binaryStr.Substring(10, 4);
                    strDEST = binaryStr.Substring(14, 9);
                    strSRC = binaryStr.Substring(23, 9);

                    if (interpretEverythingAsPASM == true)
                    {
                        // must be in PASM section
                        sb.Append("  ");
                        sb.Append(LABEL_COLUMN_MARKER);
                        sb.Append("  ");
                        sb.Append(ConvertINSRToCode(false, strINSTR, strZCRI, strCON, strDEST, strSRC));
                        sb.Append("  ");
                        return sb.ToString();
                    }

                    // should we test for the special spin sections?
                    if ((programStart != -1) && (variableBase != -1) && (stackBase != -1) && (stackStart != -1) && (programCounter >= programStart) && (programCounter <= stackStart))
                    {
                        // yes we should, are we above the program start but below
                        if ((programCounter >= programStart) && (programCounter < variableBase))
                        {
                            sb.Append("  ");
                            sb.Append(LABEL_COLUMN_MARKER);
                            sb.Append("  ");
                            sb.Append("SPIN ByteCode");
                            sb.Append("  ");
                        }
                        else if ((programCounter >= variableBase) && (programCounter < stackBase))
                        {
                            if (IsSpinStackMarker(strINSTR, strZCRI, strCON, strDEST, strSRC) == true)
                            {
                                sb.Append("  ");
                                sb.Append(LABEL_COLUMN_MARKER);
                                sb.Append("  ");
                                sb.Append("SPIN StackMarker");
                                sb.Append("  ");
                            }
                            else
                            {
                                sb.Append("  ");
                                sb.Append(LABEL_COLUMN_MARKER);
                                sb.Append("  ");
                                sb.Append("SPIN Vars");
                                sb.Append("  ");
                            }
                        }
                        else if ((programCounter >= stackBase) && (programCounter <= stackStart))
                        {
                            sb.Append("  ");
                            sb.Append(LABEL_COLUMN_MARKER);
                            sb.Append("  ");
                            sb.Append("SPIN Stack");
                            sb.Append("  ");
                        }
                    }
                    else
                    {
                        // must be in PASM section
                        sb.Append("  ");
                        sb.Append(LABEL_COLUMN_MARKER);
                        sb.Append("  ");
                        sb.Append(ConvertINSRToCode(false, strINSTR, strZCRI, strCON, strDEST, strSRC));
                        sb.Append("  ");
                    }
                }
            }

            return sb.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts 4 bytes to a spaced hex string. The four bytes are assumed to be
        /// passed in in little endian format
        /// </summary>
        /// <param name="byte0">The lowest byte to convert</param>
        /// <param name="byte1">The next byte to convert</param>
        /// <param name="byte2">The next byte to convert</param>
        /// <param name="byte3">The highest byte to convert</param>
        /// <param name="strCODE">returned value</param>
        /// <param name="strCON">returned value</param>
        /// <param name="strDEST">returned value</param>
        /// <param name="strINSTR">returned value</param>
        /// <param name="strSRC">returned value</param>
        /// <param name="strZCRI">returned value</param>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        public static void GetSplitOutPASMComponents(byte byte0, byte byte1, byte byte2, byte byte3, ref string strCODE, ref string strINSTR, ref string strZCRI, ref string strCON, ref string strDEST, ref string strSRC)
        {
            string tmpStr;
            StringBuilder sbBinary = new StringBuilder();
            string binaryStr;
            strINSTR = "";
            strZCRI = "";
            strCON = "";
            strDEST = "";
            strSRC = "";
            strCODE = "";

            // convert the byte values to binary
            tmpStr = System.Convert.ToString(byte3, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
            tmpStr = System.Convert.ToString(byte2, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
            tmpStr = System.Convert.ToString(byte1, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));
            tmpStr = System.Convert.ToString(byte0, 2);
            sbBinary.Append(PadWithLeadingZeros(tmpStr.ToUpper(), DEFAULT_BINARY_DIGITS));

            // get the various components as binary strings
            binaryStr = sbBinary.Replace(" ", "").ToString();
            strINSTR = binaryStr.Substring(0, 6);
            strZCRI = binaryStr.Substring(6, 4);
            strCON = binaryStr.Substring(10, 4);
            strDEST = binaryStr.Substring(14, 9);
            strSRC = binaryStr.Substring(23, 9);
            strCODE = ConvertINSRToCode(true, strINSTR, strZCRI, strCON, strDEST, strSRC);
            return;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts four bytes to an integer. Assumed in little endian format.
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertFourLittleEndianBytesToInteger(byte byte0, byte byte1, byte byte2, byte byte3)
        {
            // calc the int
            return (((int)(byte0)) * 1) + (((int)(byte1)) * 256) + (((int)(byte2)) * 256 * 256) + (((int)(byte3)) * 256 * 256 * 256);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts two bytes to an integer. Assumed in little endian format.
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertTwoLittleEndianBytesToInteger(byte byte0, byte byte1)
        {
            // calc the int
            return (((int)(byte0)) * 1) + (((int)(byte1)) * 256);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts one byte to an integer. 
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertOneLittleEndianByteToInteger(byte byte0)
        {
            // calc the int
            return (((int)(byte0)) * 1);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// switches things around to the binary string is in little endian 
        /// </summary>
        /// <param name="hexStr">string to process - no spaces, 8 chars</param>
        /// <returns>little endian string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static string ConvertHexStrToLittleEndian(string hexStr)
        {
            StringBuilder sb = new StringBuilder();
            if (hexStr == null) return "0";
            if (hexStr.Length != 8) return "0";

            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 6]);
            }
            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 4]);
            }
            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 2]);
            }
            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 0]);
            }
            return sb.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// switches things around to the binary string is in little endian 
        /// </summary>
        /// <param name="hexStr">little endian string to process - no spaces, 8 chars</param>
        /// <returns>little endian string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static string ConvertLittleEndianHexStrToHexStr(string hexStr)
        {
            StringBuilder sb = new StringBuilder();
            if (hexStr == null) return "0";
            if (hexStr.Length != 8) return "0";

            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 6]);
            }
            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 4]);
            }
            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 2]);
            }
            for (int i = 0; i < 2; i++)
            {
                sb.Append(hexStr[i + 0]);
            }
            return sb.ToString();
        }

       /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts an integer in absolute offset format to an integer in pasm 
        /// offset format.
        /// </summary>
        /// <param name="intToConvert">int to convert</param>
        /// <returns>compensated integer value which refers to an pasn format address</returns>
        /// <history>
        ///    16 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertIntegerInAbsoluteAddressToIntegerInPASMOffsetFormat(int intToConvert)
        {
            if (intToConvert < NUMBYTES_IN_PASM_INIT_HEADER) return 0;
            return (intToConvert - NUMBYTES_IN_PASM_INIT_HEADER) / 4;
        }
 

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// PASM stores addresses in a 9 bit field. This is the number of the 
        /// 4 byte register offset from the start of the header.
        /// </summary>
        /// <param name="intToConvert">int to convert</param>
        /// <returns>compensated integer value which refers to an absolute address</returns>
        /// <history>
        ///    16 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertIntegerInPASMAddressFormatToAbsoluteAddress(int intToConvert)
        {
            return (intToConvert * 4) + NUMBYTES_IN_PASM_INIT_HEADER;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a std format binary string to an integer. up to 32 chars
        /// </summary>
        /// <param name="binaryStr">string to process</param>
        /// <returns>binary str as an int or z for fail</returns>
        /// <history>
        ///    16 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertStdBinaryStrToInteger(string binaryStr)
        {
            try
            {
                if (binaryStr == null) return 0;
                return Convert.ToInt32(binaryStr.Replace(" ",""), 2);
            }
            catch
            {
                return 0;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a integer to a PASM hex string hence 16 goes to $10
        /// </summary>
        /// <param name="binaryStr">string to process</param>
        /// <returns>binary str as an int or z for fail</returns>
        /// <history>
        ///    16 Apr 09  Cynic - Started
        /// </history>
        public static string ConvertIntegerInPASMOffsetFormatToPASMHexStr(int intToConvert)
        {
            return "$" + System.Convert.ToString(intToConvert, 16).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a integer which is an absolute address to a hex string
        /// </summary>
        /// <param name="binaryStr">string to process</param>
        /// <returns>binary str as an int or z for fail</returns>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        public static string ConvertIntegerToHexStr(int intToConvert)
        {
            if (intToConvert < 0) return "0";
            return System.Convert.ToString(intToConvert, 16).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a string in hex format to an absolute address integer
        /// </summary>
        /// <param name="hexStr">string to process</param>
        /// <returns>integer value or zero for fail</returns>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertHexStringToInteger(string hexStr)
        {
            int workingInt = 0;

            if (hexStr == null) return 0;

            try
            {
                workingInt = Convert.ToInt32(hexStr, 16);
            }
            catch
            {
            }

            return workingInt;
        }
 
        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a hex string PASM Offset format to an absolute address integer
        /// </summary>
        /// <param name="pasmHexStr">string to process</param>
        /// <returns>absolute address or zero for fail</returns>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        public static int ConvertHexStringInPASMOffsetFormatToAbsoluteAddressAsInteger(string pasmHexStr)
        {
            int workingInt = 0;
            if (pasmHexStr == null) return 0;
            workingInt = ConvertHexStringToInteger(pasmHexStr);
            return ConvertIntegerInPASMAddressFormatToAbsoluteAddress(workingInt);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a integer to marker for a label. This is simply the hex value
        /// of the string preceeded with a LABEL_ADDRESS_MARKER char. ie "@1fd"
        /// </summary>
        /// <param name="binaryStr">string to process</param>
        /// <returns>binary str as an int or z for fail</returns>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        public static string GetStdBinaryStrAsTempLabelMarker(int intToConvert)
        {
            return LABEL_ADDRESS_MARKER + System.Convert.ToString(intToConvert, 16).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// switches things around to the binary string is in little endian 
        /// </summary>
        /// <param name="binaryStr">string to process - no spaces, 32 chars</param>
        /// <returns>little endian string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static string ConvertBinaryStrToLittleEndian(string binaryStr)
        {
            StringBuilder sb = new StringBuilder();
            if (binaryStr == null) return "0";
            if (binaryStr.Length != 32) return "0";

            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 24]);
            }
            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 16]);
            }
            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 8]);
            }
            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 0]);
            }
            return sb.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// switches things around to the binary string is in little endian 
        /// </summary>
        /// <param name="binaryStr">LittleEndian string to process - no spaces, 32 chars</param>
        /// <returns>little endian string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static string ConvertLittleEndianBinaryStrToBinaryStr(string binaryStr)
        {
            StringBuilder sb = new StringBuilder();
            if (binaryStr == null) return "0";
            if (binaryStr.Length != 32) return "0";

            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 24]);
            }
            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 16]);
            }
            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 8]);
            }
            for (int i = 0; i < 8; i++)
            {
                sb.Append(binaryStr[i + 0]);
            }
            return sb.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Pads out the string with zeros so it reaches the specified length
        /// </summary>
        /// <param name="dist">character distance between separators</param>
        /// <param name="tmpStr">string to process</param>
        /// <returns>padded out string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static string InsertSpaceSeparators(string tmpStr, int dist)
        {
            if (tmpStr == null) return "0";
            StringBuilder sb = new StringBuilder();

            // add in the padding
            for (int i = 0; i < tmpStr.Length; i++)
            {
                if (((i % dist) == 0) && i != 0) sb.Append(" ");
                sb.Append(tmpStr[i]);
            }
            return sb.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Tests to see if the input PASM components are a SPIN command which
        /// looks like a PASM NOP
        /// </summary>
        /// <param name="inStr">INSTR string to process - no spaces, 6 chars</param>
        /// <param name="zcriStr">ZCRI string to process - no spaces, 4 chars</param>
        /// <param name="conStr">CON string to process - no spaces, 4 chars</param>
        /// <param name="destStr">DEST string to process - no spaces, 9 chars</param>
        /// <param name="srcStr">SRC string to process - no spaces, 9 chars</param>
        /// <returns>code string</returns>
        /// <history>
        ///    11 Apr 09  Cynic - Started
        /// </history>
        private static bool IsSpinNOP(string inStr, string zcriStr, string conStr, string destStr, string srcStr)
        {
            if (inStr == null) return false;
            if (inStr.Length != 6) return false;
            if (zcriStr == null) return false;
            if (zcriStr.Length != 4) return false;
            if (conStr == null) return false;
            if (conStr.Length != 4) return false;
            if (destStr == null) return false;
            if (destStr.Length != 9) return false;
            if (srcStr == null) return false;
            if (srcStr.Length != 9) return false;

            // this is what determines a NOP
            if (conStr == "0000")
            {
                // perform checks for possible spin code variants
                if (srcStr != "000000000") return true;
                if (destStr != "000000000") return true;
                if (conStr != "0000") return true;
                if (zcriStr != "0000") return true;
                if (inStr != "000000") return true;
                // looks like a genuine PASM NOP 
                return false;
            }
            return false;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Tests to see if the input PASM components are a SPIN command which
        /// looks like a PASM NOP
        /// </summary>
        /// <param name="inStr">INSTR string to process - no spaces, 6 chars</param>
        /// <param name="zcriStr">ZCRI string to process - no spaces, 4 chars</param>
        /// <param name="conStr">CON string to process - no spaces, 4 chars</param>
        /// <param name="destStr">DEST string to process - no spaces, 9 chars</param>
        /// <param name="srcStr">SRC string to process - no spaces, 9 chars</param>
        /// <returns>code string</returns>
        /// <history>
        ///    11 Apr 09  Cynic - Started
        /// </history>
        private static bool IsSpinStackMarker(string inStr, string zcriStr, string conStr, string destStr, string srcStr)
        {
            if (inStr == null) return false;
            if (inStr.Length != 6) return false;
            if (zcriStr == null) return false;
            if (zcriStr.Length != 4) return false;
            if (conStr == null) return false;
            if (conStr.Length != 4) return false;
            if (destStr == null) return false;
            if (destStr.Length != 9) return false;
            if (srcStr == null) return false;
            if (srcStr.Length != 9) return false;

            // special case have we got a spin VAR/STACK Marker
            if ((srcStr == "111111111") &&
               (destStr == "011111111") &&
               (conStr == "1110") &&
               (zcriStr == "1111") &&
               (inStr == "111111"))
            {
                return true;
            }
            return false;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a six char INSTR binary value to an alphanumeric hex value
        /// </summary>
        /// <param name="opCodeOnly">if true we do not return the operand text</param>
        /// <param name="inStr">INSTR string to process - no spaces, 6 chars</param>
        /// <param name="zcriStr">ZCRI string to process - no spaces, 4 chars</param>
        /// <param name="conStr">CON string to process - no spaces, 4 chars</param>
        /// <param name="destStr">DEST string to process - no spaces, 9 chars</param>
        /// <param name="srcStr">SRC string to process - no spaces, 9 chars</param>
        /// <returns>code string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private static string ConvertINSRToCode(bool opCodeOnly, string inStr, string zcriStr, string conStr, string destStr, string srcStr)
        {

            string opCode = "";
            string conOpCode = "";
            string opCodeOperands = "";

            if (inStr == null) return "?";
            if (inStr.Length != NUM_BITS_IN_OPCODE_INSTR) return "??";
            if (zcriStr == null) return "?";
            if (zcriStr.Length != NUM_BITS_IN_OPCODE_ZCRI) return "??";
            if (conStr == null) return "?";
            if (conStr.Length != NUM_BITS_IN_OPCODE_CON) return "??";
            if (destStr == null) return "?";
            if (destStr.Length != NUM_BITS_IN_OPCODE_DEST) return "??";
            if (srcStr == null) return "?";
            if (srcStr.Length != NUM_BITS_IN_OPCODE_SRC) return "??";

            // special case have we got a spin VAR/STACK Marker
            if(IsSpinStackMarker(inStr, zcriStr, conStr, destStr, srcStr)==true)
            {
                return "SPIN StackMarker";
            }

            // this is what determines a NOP
            if (conStr == "0000")
            {
                if (IsSpinNOP(inStr, zcriStr, conStr, destStr, srcStr) == true)
                {
                    return "SPIN?";
                }
                else
                {
                    // looks like a genuine PASM NOP 
                    return "NOP";
                }
            }

            // convert the condition opcode now
            conOpCode = GetConOpCode(conStr);

            // now figure out what instruction we are dealing with based on the INSTR
            switch (inStr)
            {
                case "101010":
                    opCode = "ABS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true); ;
                    break;
                case "101011":
                    opCode = "ABSNEG";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "100000":
                    opCode = "ADD";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "100010":
                    opCode = "ADDABS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "110100":
                    opCode = "ADDS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "110110":
                    opCode = "ADDSX";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "110010":
                    opCode = "ADDX";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "011000":
                    if (zcriStr[2] == '1') 
                    {
                        opCode = "AND";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                        break;
                    }
                    else 
                    {
                        opCode = "TEST";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, true, true, false);
                        break;
                    }
                case "011001":
                    opCode = "ANDN";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010111":
                    // is the R bit high - this means CALL or JMPRET
                    if (zcriStr[2] == '1') 
                    {
                        // is the i bit high - this usually means a CALL (but could also be a JMPRET with absolute 
                        // which is the same thing)
                        if (zcriStr[3] == '1')
                        {
                            // the 
                            opCode = "CALL";
                            opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, true, false, true, false, false, true, false, false, true);
                            break;
                        }
                        else
                        {
                            opCode = "JMPRET";
                            opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                            break;
                        }
                    }
                    else 
                    {
                        // R bit low means a jump or ret
                        if (srcStr == "000000000")
                        {
                            // this value usually means it is a RET with a dynamiclly filled in SRC from
                            // a call or a JMPRET. Otherwise it is just a jump to address $0. We assume
                            // for clarity that this is a RET with a dynamic src field
                            opCode = "RET <DynamicTarget>";
                            opCodeOperands = "";
                            break;
                        }
                        else
                        {
                            // assume a forced jump to a label
                            opCode = "JMP";
                            opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, true, false, true, false, false, true, false, false, false);
                            break;
                        }
                    }
                case "000011":
                    if (srcStr.EndsWith("000") == true) 
                    {
                        opCode = "CLKSET";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, true, false, false, false);
                        break;
                    }
                    else if (srcStr.EndsWith("001") == true)
                    {
                        opCode = "COGID";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, false, false, false, true);
                        break;
                    }
                    else if (srcStr.EndsWith("010") == true) 
                    {
                        opCode = "COGINIT";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, false, false, false, false);
                        break;
                    }
                    else if (srcStr.EndsWith("011") == true) 
                    {
                        opCode = "COGSTOP";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, false, false, false, false);
                        break;
                    }
                    else if (srcStr.EndsWith("111") == true) 
                    {
                        opCode = "LOCKCLR";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, false, false, false, false);
                        break;
                    }
                    else if (srcStr.EndsWith("100") == true) 
                    {
                        opCode = "LOCKNEW";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, false, false, false, true);
                        break;
                    }
                    else if (srcStr.EndsWith("101") == true) 
                    {
                        opCode = "LOCKRET";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, false, false, false, false);
                        break;
                    }
                    else if (srcStr.EndsWith("110") == true) 
                    {
                        opCode = "LOCKSET";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, true, false, false, false, false, false, false, false);
                        break;
                    }
                    else 
                    {
                        opCode = "?";
                        break;
                    }
                case "100001":
                    // is the R bit high - this means SUB
                    if (zcriStr[2] == '1') 
                    {
                        opCode = "SUB";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                        break;
                    }
                    else
                    {
                        opCode = "CMP";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                        break;
                    }
                case "110000":
                    opCode = "CMPS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false); 
                    break;
                case "111000":
                    opCode = "CMPSUB";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true); 
                    break;
                case "110001":
                    opCode = "CMPSX";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                    break;
                case "110011":
                    // is the R bit high - this means SUB
                    if (zcriStr[2] == '1')
                    {
                        opCode = "SUBX";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                        break;
                    }
                    else
                    {
                        opCode = "CMPX";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                        break;
                    }
                case "111001":
                    opCode = "DJNZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010011":
                    opCode = "MAX";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010001":
                    opCode = "MAXS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010010":
                    opCode = "MIN";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010000":
                    opCode = "MINS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "101000":
                    opCode = "MOV";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010101":
                    opCode = "MOVD";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010110":
                    opCode = "MOVI";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "010100":
                    opCode = "MOVS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "011100":
                    opCode = "MUXC";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "011101":
                    opCode = "MUXNC";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "011111":
                    opCode = "MUXNZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "011110":
                    opCode = "MUXZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "101001":
                    opCode = "NEG";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "101100":
                    opCode = "NEGC";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "101101":
                    opCode = "NEGNC";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "101111":
                    opCode = "NEGNZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "101110":
                    opCode = "NEGZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "011010":
                    opCode = "OR";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "001101":
                    opCode = "RCL";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "001100":
                    opCode = "RCR";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "000000":
                    // is the R bit high - this means RDBYTE
                    if (zcriStr[2] == '1')
                    {
                        opCode = "RDBYTE";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, true, false, false, false);
                        break;
                    }
                    else 
                    {
                        opCode = "WRBYTE";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, true, false, false, false);
                        break;
                    }
                case "000010":
                    // is the R bit high - this means RDLONG
                    if (zcriStr[2] == '1')
                    {
                        opCode = "RDLONG";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, true, false, false, false);
                        break;
                    }
                    else
                    {
                        opCode = "WRLONG";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, true, false, false, false);
                        break;
                    }
                case "000001":
                    // is the R bit high - this means RDWORD
                    if (zcriStr[2] == '1')
                    {
                        opCode = "RDWORD";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, true, false, false, false);
                        break;
                    }
                    else
                    {
                        opCode = "WRWORD";
                        opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, true, false, false, false);
                        break;
                    }
                case "001111":
                    opCode = "REV";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "001001":
                    opCode = "ROL";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "001000":
                    opCode = "ROR";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "001110":
                    opCode = "SAR";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "001011":
                    opCode = "SHL";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "001010":
                    opCode = "SHR";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "100011":
                    opCode = "SUBABS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "110101":
                    opCode = "SUBS";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "110111":
                    opCode = "SUBSX";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "100100":
                    opCode = "SUMC";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "100101":
                    opCode = "SUMNC";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "100111":
                    opCode = "SUMNZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "100110":
                    opCode = "SUMZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "111010":
                    opCode = "TJNZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                    break;
                case "111011":
                    opCode = "TJZ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                    break;
                case "111110":
                    opCode = "WAITCNT";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                case "111100":
                    opCode = "WAITPEQ";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                    break;
                case "111101":
                    opCode = "WAITPNE";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                    break;
                case "111111":
                    opCode = "WAITVID";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
                    break;
                case "011011":
                    opCode = "XOR";
                    opCodeOperands = GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, true);
                    break;
                default:
                    opCode = "?";
                    break;
            } // bottom of switch

            StringBuilder retStr = new StringBuilder();

            // do we have conOpcode
            if ((conOpCode.Length != 0) && (opCodeOnly == false))
            {
                retStr.Append(conOpCode);
                retStr.Append(" ");
            }

            // add this now
            retStr.Append(opCode);

            // do we have opCodeOperands
            if ((opCodeOperands.Length != 0) && (opCodeOnly == false))
            {
                retStr.Append(" ");
                retStr.Append(opCodeOperands);
            }

            return retStr.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts a binary Condition string to an opcode 
        /// </summary>
        /// <param name="conStr">condition string</param>
        /// <returns>code string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private static string GetConOpCode(string conStr)
        {
            string outStr = "";

            if (conStr == null) return "";
            if (conStr.Length != NUM_BITS_IN_OPCODE_CON) return "";

            // now switch and return the condition opcode
            switch (conStr)
            {
                case "0000":
                    outStr = "IF_NEVER";
                    break;
                case "0001":
                    outStr = "IF_A";
                    break;
                case "0010":
                    outStr = "IF_NC_AND_Z";
                    break;
                case "0011":
                    outStr = "IF_NC";
                    break;
                case "0100":
                    outStr = "IF_C_AND_NZ";
                    break;
                case "0101":
                    outStr = "IF_NZ";
                    break;
                case "0110":
                    outStr = "IF_C_NE_Z";
                    break;
                case "0111":
                    outStr = "IF_NC_OR_NZ";
                    break;
                case "1000":
                    outStr = "IF_C_AND_Z";
                    break;
                case "1001":
                    outStr = "IF_C_EQ_Z";
                    break;
                case "1010":
                    outStr = "IF_Z";
                    break;
                case "1011":
                    outStr = "IF_NC_OR_Z";
                    break;
                case "1100":
                    outStr = "IF_C";
                    break;
                case "1101":
                    outStr = "IF_NZ_OR_C";
                    break;
                case "1110":
                    outStr = "IF_C_OR_Z";
                    break;
                case "1111":
                    //outStr = "IF_ALWAYS";
                    // fall through, use a "" for IF_ALWAYS
                default:
                    outStr = "";
                    break;
            }

            return outStr;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the subsidiary code which follows an OPCODE - essentially the rest
        /// of the dissasembled line. This is the standard version which many opcodes
        /// use. The dest is always an address, and the src is either an address
        /// or a 
        /// </summary>
        /// <param name="inStr">INSTR string to process - no spaces, 6 chars</param>
        /// <param name="zcriStr">ZCRI string to process - no spaces, 4 chars</param>
        /// <param name="conStr">CON string to process - no spaces, 4 chars</param>
        /// <param name="destStr">DEST string to process - no spaces, 9 chars</param>
        /// <param name="srcStr">SRC string to process - no spaces, 9 chars</param>
        /// <param name="suppressWC">if true suppress reporting of the WC flag</param>
        /// <param name="suppressWZ">if true suppress reporting of the WZ flag</param>
        /// <param name="suppressWR">if true suppress reporting of the WR flag</param>
        /// <param name="defaultHighWC">if true defaultHigh reporting of the WC flag to a NC</param>
        /// <param name="defaultHighWZ">if true defaultHigh reporting of the WZ flag to a NZ</param>
        /// <param name="defaultHighWR">if true defaultHigh reporting of the WR flag to a NR</param>
        /// <param name="ignoreDest">if true we ignore the dest and do not output its operand</param>
        /// <param name="ignoreSrc">if true we ignore the src and do not output its operand</param>
        /// <param name="srcIsAlwaysLabel">Treat src as a PASM label even if zcri is not immediate</param>
        /// <returns>code string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private static string GetStandardOperands(string inStr, string zcriStr, string conStr, string destStr, string srcStr, bool ignoreDest, bool ignoreSrc, bool srcIsAlwaysLabel, bool suppressWZ, bool suppressWC, bool suppressWR, bool defaultHighWZ, bool defaultHighWC, bool defaultHighWR)
        {
            StringBuilder sbFlags = new StringBuilder();
            StringBuilder sbDest = new StringBuilder();
            StringBuilder sbSrc = new StringBuilder();
            StringBuilder sbOut = new StringBuilder();
            string tmpStr=null;

            if ((inStr == null) || (inStr.Length != NUM_BITS_IN_OPCODE_INSTR)) return "?";
            if ((zcriStr == null) || (zcriStr.Length != NUM_BITS_IN_OPCODE_ZCRI)) return "?";
            if ((conStr == null) || (conStr.Length != NUM_BITS_IN_OPCODE_CON)) return "?";
            if ((destStr == null) || (destStr.Length != NUM_BITS_IN_OPCODE_DEST)) return "?";
            if ((srcStr == null) || (srcStr.Length != NUM_BITS_IN_OPCODE_SRC)) return "?";

            // do we want to report the Z flag
            if(suppressWZ == false)
            {
                if ((zcriStr[0] == '0') && (defaultHighWZ == true))
                {
                    if (sbFlags.Length != 0) sbFlags.Append(", ");
                    sbFlags.Append("NZ");
                }
                else if ((zcriStr[0] == '1') && (defaultHighWZ == false))
                {
                    if (sbFlags.Length != 0) sbFlags.Append(", ");
                    sbFlags.Append("WZ");
                }
            }

            // do we want to report the C flag
            if(suppressWC==false)
            {
                if ((zcriStr[1] == '0') && (defaultHighWC == true))
                {
                    if (sbFlags.Length != 0) sbFlags.Append(", ");
                    sbFlags.Append("NC");
                }
                else if ((zcriStr[1] == '1') && (defaultHighWC == false))
                {
                    if (sbFlags.Length != 0) sbFlags.Append(", ");
                    sbFlags.Append("WC");
                }
            }

            // do we want to report the R flag
            if(suppressWR==false)
            {
                if ((zcriStr[2] == '0') && (defaultHighWR == true))
                {
                    if (sbFlags.Length != 0) sbFlags.Append(", ");
                    sbFlags.Append("NR");
                }
                else if ((zcriStr[2] == '1') && (defaultHighWR == false))
                {
                    if (sbFlags.Length != 0) sbFlags.Append(", ");
                    sbFlags.Append("WR");
                }
            }

            // now do the dest
            tmpStr = GetAddressBinaryStrAsKnownRegister(destStr);
            if ((tmpStr == null) || (tmpStr.Length == 0))
            {
                // append a label marker, we fix this up later according to the options
                // set by the user
                sbDest.Append(GetStdBinaryStrAsTempLabelMarker(ConvertStdBinaryStrToInteger(destStr)));
            }
            else
            {
                // it is a known register append the text
                sbDest.Append(tmpStr);
            }

            // now do the src field, are we immediate?
            if ((zcriStr[3] == '0') || (srcIsAlwaysLabel == true))
            {
                // no we are not immediate
                tmpStr = GetAddressBinaryStrAsKnownRegister(srcStr);
                if ((tmpStr == null) || (tmpStr.Length == 0))
                {
                    // this means src is an address append a label marker, 
                    // we fix this up later according to the options set by the user
                    sbSrc.Append(GetStdBinaryStrAsTempLabelMarker(ConvertStdBinaryStrToInteger(srcStr)));
                }
                else
                {
                    // it is a known register append the text
                    sbSrc.Append(tmpStr);
                }
            }
            else
            {
                // this means src is immediate
                tmpStr = GetAddressBinaryStrAsKnownRegister(srcStr);
                if ((tmpStr == null) || (tmpStr.Length == 0))
                {
                    // since we are an immediate value, flag it now
                    sbSrc.Append("#");
                    // this means it is not a known register hence must be a simple number
                    sbSrc.Append(ConvertIntegerInPASMOffsetFormatToPASMHexStr(ConvertStdBinaryStrToInteger(srcStr)));
                }
                else
                {
                    // since we are an immediate value, flag it now
                    sbSrc.Append("#");
                    // it is a known register append the text
                    sbSrc.Append(tmpStr);
                }
            }

            // build the output string
            if((sbDest.Length!=0) && (ignoreDest==false))
            {
                sbOut.Append(sbDest);
            }
            if((sbSrc.Length!=0) && (ignoreSrc==false))
            {
                sbOut.Append(" ");
                sbOut.Append(sbSrc);
            }
            if(sbFlags.Length!=0)
            {
                sbOut.Append(" ");
                sbOut.Append(sbFlags);
            }
            return  sbOut.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the subsidiary code which follows an OPCODE - essentially the rest
        /// of the dissasembled line. This is the standard version which many opcodes
        /// use. The dest is always an address, and the src is either an address
        /// or a 
        /// </summary>
        /// <param name="inStr">INSTR string to process - no spaces, 6 chars</param>
        /// <param name="zcriStr">ZCRI string to process - no spaces, 4 chars</param>
        /// <param name="conStr">CON string to process - no spaces, 4 chars</param>
        /// <param name="destStr">DEST string to process - no spaces, 9 chars</param>
        /// <param name="srcStr">SRC string to process - no spaces, 9 chars</param>
        /// <returns>code string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private static string GetStandardOperands(string inStr, string zcriStr, string conStr, string destStr, string srcStr)
        {
            return GetStandardOperands(inStr, zcriStr, conStr, destStr, srcStr, false, false, false, false, false, false, false, false, false);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Accepts a binary str value and if it matches one of our special register
        /// we return the text for that register
        /// </summary>
        /// <param name="strValue">the value to check</param>
        /// <returns>register name or "" for fail or not known</returns>
        /// <history>
        ///    16 Apr 09  Cynic - Started
        /// </history>
        public static string GetAddressBinaryStrAsKnownRegister(string binaryStr)
        {
            try
            {
                if (binaryStr == null) return "";
                int workingInt = ConvertStdBinaryStrToInteger(binaryStr);
                return GetAddressIntAsKnownRegister(workingInt);
            }
            catch
            {
                return "";
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Accepts an int value and if it matches one of our special register
        /// we return the text for that register
        /// </summary>
        /// <param name="addressInt">an int in PASM address format</param>
        /// <returns>register name or "" for fail or not known</returns>
        /// <history>
        ///    16 Apr 09  Cynic - Started
        /// </history>
        public static string GetAddressIntAsKnownRegister(int addressInt)
        {
            switch (addressInt)
            {
                case 0x1F0:
                    return "PAR";
                case 0x1F1:
                    return "CNT";
                case 0x1F2:
                    return "INA";
                case 0x1F3:
                    return "INB";
                case 0x1F4:
                    return "OUTA";
                case 0x1F5:
                    return "OUTB";
                case 0x1F6:
                    return "DIRA";
                case 0x1F7:
                    return "DIRB";
                case 0x1F8:
                    return "CTRA";
                case 0x1F9:
                    return "CTRB";
                case 0x1FA:
                    return "FRQA";
                case 0x1FB:
                    return "FRQB";
                case 0x1FC:
                    return "PHSA";
                case 0x1FD:
                    return "PHSB";
                case 0x1FE:
                    return "VCFG";
                case 0x1FF:
                    return "VSCL";
                default:
                    return "";
            }
        }
                            

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Pads out the string with zeros so it reaches the specified length
        /// </summary>
        /// <param name="len">total desired length of string</param>
        /// <param name="tmpStr">string to process</param>
        /// <returns>padded out string</returns>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public static string PadWithLeadingZeros(string tmpStr, int len)
        {
            if (tmpStr == null) return "0";
            StringBuilder sb = new StringBuilder();

            int numZerosToAdd = len - tmpStr.Length;
            if (numZerosToAdd <= 0) return tmpStr;

            // add on the zeros
            for (int i = 0; i < numZerosToAdd; i++)
            {
                sb.Append("0");
            }
            return sb.ToString() + tmpStr;
        }

    }
}
