using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

/// +------------------------------------------------------------------------------------------------------------------------------+
///                                                    TERMS OF USE: MIT License                                                  
/// +------------------------------------------------------------------------------------------------------------------------------
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    
/// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
/// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
/// is furnished to do so, subject to the following conditions:                                                                   
///                                                                                                                               
/// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
///                                                                                                                               
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
/// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
/// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
/// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
/// +------------------------------------------------------------------------------------------------------------------------------+

namespace PropScriptBitInspector
{
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// <summary>
    /// A class to read a binary file from disk and populate a RichTextBox
    /// with the output in a nice display format
    /// </summary>
    /// <history>
    ///    10 Apr 09  Cynic - Started
    /// </history>
    public class BinaryFileFormatter
    {
        public const bool DEFAULT_WANT_PROGRAM_COUNTER = true;
        private bool wantProgramCounter = DEFAULT_WANT_PROGRAM_COUNTER;
        public const bool DEFAULT_WANT_BINARY_VALUES = false;
        private bool wantBinaryValues = DEFAULT_WANT_BINARY_VALUES;
        public const bool DEFAULT_REMOVE_LITTLE_ENDIAN = false;
        private bool removeLittleEndian = DEFAULT_REMOVE_LITTLE_ENDIAN;
        public const bool DEFAULT_WANT_PASM_CODES = true;
        private bool wantPasmCodes = DEFAULT_WANT_PASM_CODES;

        public const bool DEFAULT_DISPLAY_LABELS_AS_TEXT = true;
        private bool displayLabelsAsText = DEFAULT_DISPLAY_LABELS_AS_TEXT;
        public const bool DEFAULT_DISPLAY_LABEL_HEX_AS_ABSOLUTE = true;
        private bool displayLabelHexAsAbsolute = DEFAULT_DISPLAY_LABEL_HEX_AS_ABSOLUTE;
        public const bool DEFAULT_INTERPRET_EVERYTHING_AS_PASM = false;
        private bool interpretEverythingAsPASM = DEFAULT_INTERPRET_EVERYTHING_AS_PASM;

        private string binaryFileName = null;
        private byte [] binaryFileInMemory=null;

        // we use this for label fixups
        private string[] labelArray = new string[BinaryDataConverter.MAX_LONGS_IN_COGMEMORY + 2];

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the name of the current binary file. There is no set accessor
        /// the name is set on the call to ReadFileIntoMemory
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string BinaryFileName
        {
            get
            {
                if (binaryFileName == null) return "";
                return binaryFileName;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets wether we want the program counter line
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public bool WantProgramCounter
        {
            get
            {
                return wantProgramCounter;
            }
            set
            {
                wantProgramCounter = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets wether we want the binary values lines
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public bool WantBinaryValues
        {
            get
            {
                return wantBinaryValues;
            }
            set
            {
                wantBinaryValues = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets wether we want the PASM codes
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public bool WantPasmCodes
        {
            get
            {
                return wantPasmCodes;
            }
            set
            {
                wantPasmCodes = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets wether we want to remove little endian ness from tehfile
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public bool RemoveLittleEndian
        {
            get
            {
                return removeLittleEndian;
            }
            set
            {
                removeLittleEndian = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets wether we want to display labels as text
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - originally written
        /// </history>
        public bool DisplayLabelsAsText
        {
            get
            {
                return displayLabelsAsText;
            }
            set
            {
                displayLabelsAsText = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets wether we want to display labels hex as absolute addresses
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - originally written
        /// </history>
        public bool DisplayLabelHexAsAbsolute
        {
            get
            {
                return displayLabelHexAsAbsolute;
            }
            set
            {
                displayLabelHexAsAbsolute = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets wether we want to display labels hex as absolute addresses
        /// </summary>
        /// <history>
        ///    26 Apr 09  Cynic - originally written
        /// </history>
        public bool InterpretEverythingAsPASM
        {
            get
            {
                return interpretEverythingAsPASM;
            }
            set
            {
                interpretEverythingAsPASM = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Reads an (assumed) binary file from disk and stores it in our byte array
        /// </summary>
        /// <param name="fileToOpen">name of file to open</param>
        /// <param name="errStr">we return any errors in here</param>
        /// <returns>z success, nz faile</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public int ReadFileIntoMemory(string fileToOpen, ref string errStr)
        {
            int fileLen=0;
            BinaryReader bReader = null;
            FileStream fStream = null;

            errStr = "";
            if (fileToOpen == null)
            {
                errStr = "fileToOpen == null";
                return 101;
            }

            if (File.Exists(fileToOpen) == false)
            {
                errStr = "File does not exist: " + fileToOpen;
                return 102;
            }

            try
            {

                // we need this to get the size of the file
                FileInfo fInfo = new FileInfo(fileToOpen);

                fileLen = (int)fInfo.Length;

                // get a FileStream
                fStream = new FileStream(fileToOpen, FileMode.Open, FileAccess.Read);
                if (fStream == null)
                {
                    errStr = "fStream == null";
                    return 103;
                }

                // get a BinaryReader
                bReader = new BinaryReader(fStream);
                if (bReader == null)
                {
                    errStr = "bReader == null";
                    return 104;
                }

                // read the binary file into memory,
                binaryFileInMemory = bReader.ReadBytes(fileLen);
                if (binaryFileInMemory == null)
                {
                    errStr = "binaryFileInMemory == null";
                    return 105;
                }

                // set this now
                binaryFileName = fileToOpen;
            }
            catch (Exception ex)
            {
                errStr = "exception: " + ex.Message;
                return 201;
            }
            finally
            {
                if (bReader != null) bReader.Close();
                if (fStream!=null) fStream.Close();
            }

            return 0;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the bytes at an offset
        /// </summary>
        /// <returns>the bytes</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public void GetBytesAtOffset(int offset, ref byte byte0, ref byte byte1, ref byte byte2, ref byte byte3)
        {
            byte0 = 0;
            byte1 = 0;
            byte2 = 0;
            byte3 = 0;
            if (binaryFileInMemory == null) return;
            if (binaryFileInMemory.Length<=offset+3) return;
            byte0 = binaryFileInMemory[offset + 0];
            byte1 = binaryFileInMemory[offset + 1];
            byte2 = binaryFileInMemory[offset + 2];
            byte3 = binaryFileInMemory[offset + 3];
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Reads the byte array and formats it up as a display string. The file must
        /// have been previously open;
        /// </summary>
        /// <returns>the display text as a string</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetDisplayText()
        {
            int offsetAddress = 0;
            string destAddr=null;
            string srcAddr=null;

            List<string>lineArray = new List<string>();
            StringBuilder sb = new StringBuilder();
            string workingString = null;

            if(binaryFileInMemory==null) return "";
            for(int i=0;i<binaryFileInMemory.Length;i+=4)
            {
                workingString = BinaryDataConverter.GetBinaryFileDisplayLine(InterpretEverythingAsPASM, RemoveLittleEndian, binaryFileInMemory[i], binaryFileInMemory[i + 1], binaryFileInMemory[i + 2], binaryFileInMemory[i + 3], WantProgramCounter, i, wantBinaryValues, WantPasmCodes, GetInitialProgramCounterAsInt(), GetVariableBaseAsInt(), GetStackBaseAsInt(), GetInitialStackCounterAsInt());
                lineArray.Add(workingString+"\r\n");
            }

            // now we fix up labels according to the users requirements
            // first we troll through the list looking for references to
            // labels in the pasm code
            for (int i = 0; i < lineArray.Count; i++)
            {
                // look for any src or dest fields that reference a label
                GetDestAndSrcLabelAddrFromString(lineArray[i], ref destAddr, ref srcAddr);
                // record any labels that came back, these can be null
                RecordLabel(destAddr);
                RecordLabel(srcAddr);
            }

            // now make the labels look nice.
            MakeLabelsLookHumanReadable();

            // now go back and adjust all strings. remove the LABEL_COLUMN_MARKER values
            // for lines which do not have a label and replace the LABEL_COLUMN_MARKER
            // with appropriate labels for those which do
            for (int i = 0; i < lineArray.Count; i++)
            {
                // this is the actual address of the start of this lines binary bytes
                offsetAddress = i * 4;
                if (offsetAddress >= BinaryDataConverter.MAX_LONGS_IN_COGMEMORY)
                {
                    // we could never have a labelf or this - remove the label marker
                    lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_COLUMN_MARKER, BinaryDataConverter.LABEL_COLUMN_MARKER_BLANK);
                    continue;
                }

                // do we have a label recorded for this offset?
                if ((labelArray[offsetAddress] == null) || (labelArray[offsetAddress].Length == 0))
                {
                    // no, we do not. Remove the label column marker
                    lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_COLUMN_MARKER, BinaryDataConverter.LABEL_COLUMN_MARKER_BLANK);
                }
                else
                {
                    // yes we do replace the label column marker with the fixed up label
                    lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_COLUMN_MARKER, labelArray[offsetAddress]);
                }

                // now we go back and fixup the dest and src labels (if any)
                GetDestAndSrcLabelAddrFromString(lineArray[i], ref destAddr, ref srcAddr);
                if ((destAddr != null) && (destAddr.Length > 0))
                {
                    offsetAddress = BinaryDataConverter.ConvertHexStringInPASMOffsetFormatToAbsoluteAddressAsInteger(destAddr);
                    if (offsetAddress < BinaryDataConverter.MAX_LONGS_IN_COGMEMORY)
                    {
                        if ((labelArray[offsetAddress] != null) && (labelArray[offsetAddress].Length > 0))
                        {
                            // replace the value
                            lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_ADDRESS_MARKER + destAddr, labelArray[offsetAddress]);
                        }
                        else
                        {
                            // should never happen implies bug in code
                            lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_ADDRESS_MARKER + destAddr, BinaryDataConverter.LABEL_ADDRESS_MARKER + "$" + destAddr);
                        }
                    }
                    else
                    {
                        // means user entered an constant but did not specify the # label and the constant is out of range
                        lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_ADDRESS_MARKER + destAddr, BinaryDataConverter.LABEL_ADDRESS_MARKER + "$" + destAddr + " (badaddr->$" + BinaryDataConverter.ConvertIntegerToHexStr(offsetAddress) + ")");
                    }
                }
                if ((srcAddr != null) && (srcAddr.Length > 0))
                {
                    offsetAddress = BinaryDataConverter.ConvertHexStringInPASMOffsetFormatToAbsoluteAddressAsInteger(srcAddr);
                    if (offsetAddress < BinaryDataConverter.MAX_LONGS_IN_COGMEMORY)
                    {
                        if ((labelArray[offsetAddress] != null) && (labelArray[offsetAddress].Length > 0))
                        {
                            // replace the value
                            lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_ADDRESS_MARKER + srcAddr, labelArray[offsetAddress]);
                        }
                        else
                        {
                            // should never happen implies bug in code
                            lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_ADDRESS_MARKER + srcAddr, BinaryDataConverter.LABEL_ADDRESS_MARKER + "$" + srcAddr);
                        }
                    }
                    else
                    {
                        // means user entered an constant but did not specify the # label and the constant is out of range
                        lineArray[i] = lineArray[i].Replace(BinaryDataConverter.LABEL_ADDRESS_MARKER + srcAddr, BinaryDataConverter.LABEL_ADDRESS_MARKER + "$" + srcAddr + " (badaddr->$" + BinaryDataConverter.ConvertIntegerToHexStr(offsetAddress) + ")");
                    }
                }
            }


            // now build a single string of it and return the lot
            for (int i = 0; i < lineArray.Count; i ++)
            {
                sb.Append(lineArray[i]);
            }
            return sb.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Make the labels look nice. This essentially means taking the PASM offset
        /// address and appending a "L_" to it
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - originally written
        /// </history>
        private void MakeLabelsLookHumanReadable()
        {
            StringBuilder sbLabelStr = new StringBuilder();
            // for each entry in the label array, fix up the entry so it is nice looking
            for (int i = 0; i < BinaryDataConverter.MAX_LONGS_IN_COGMEMORY; i++)
            {
                // have we got an entry?
                if ((labelArray[i] == null) || (labelArray[i].Length == 0)) continue;
                if (this.DisplayLabelsAsText == true)
                {
                    // yes we have, build the next label
                    if (labelArray[i].Length == 1) labelArray[i] = "L__" + labelArray[i];
                    else if (labelArray[i].Length == 2) labelArray[i] = "L_" + labelArray[i];
                    else if (labelArray[i].Length == 13) labelArray[i] = "L" + labelArray[i];
                    else
                    {
                        // just leave it as-is
                    }
                }
                else
                {
                    // user requested to leave the labels as hex
                    if (this.DisplayLabelHexAsAbsolute == false)
                    {
                        // just leave as-is - pad with zeros
                        labelArray[i] = BinaryDataConverter.PadWithLeadingZeros(labelArray[i], 4);
                    }
                    else
                    {
                        // convert to absolute address and pad with zeros
                        int workingInt = BinaryDataConverter.ConvertHexStringToInteger(labelArray[i]);
                        workingInt = BinaryDataConverter.ConvertIntegerInPASMAddressFormatToAbsoluteAddress(workingInt);
                        string workingStr = BinaryDataConverter.ConvertIntegerToHexStr(workingInt);
                        labelArray[i] = BinaryDataConverter.PadWithLeadingZeros(workingStr, 4);
                    }
                }
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Accepts a string and generates a label entry for it. The string
        /// is assumed to be a hexadecimal address as a string in PASM offset format
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - originally written
        /// </history>
        private void RecordLabel(string labelIn)
        {
            if (labelIn == null) return;
            if (labelIn.Length == 0) return;

            int offsetAddr = BinaryDataConverter.ConvertHexStringInPASMOffsetFormatToAbsoluteAddressAsInteger(labelIn);
            if (offsetAddr < 0) return;
            if ((offsetAddr) >= BinaryDataConverter.MAX_LONGS_IN_COGMEMORY) return;

            // ok, the integer offset address represents a long which is the target of 
            // a label. We record this in the labelArray. We just stuff in the labelIn
            // value - the actual value does not matter - just the fact that a non null
            // value is present at address labelArray[offsetAddr] is sufficient to mark
            // the label.
            try
            {
                labelArray[offsetAddr] = labelIn;
            }
            catch (Exception ex)
            {
                string foo= ex.Message;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Looks in a string, finds any references to a BinaryDataConverter.LABEL_ADDRESS_MARKER
        /// and parses out the contents. The first is returned as the destAddr and the second as
        /// the src addr
        /// </summary>
        /// <param name="srcAddr">we return the second label addres here</param>
        /// <param name="destAddr">we return the first label addres here</param>
        /// <param name="inputStr">the input string we look in</param>
        /// <returns>z no labelAddrs found, nz at least one labelAddr was found</returns>
        /// <history>
        ///    18 Apr 09  Cynic - originally written
        /// </history>
        private int GetDestAndSrcLabelAddrFromString(string inputStr, ref string destAddr, ref string srcAddr)
        {
            StringBuilder sbAddr = new StringBuilder();
            bool inAddrNow=false;
            destAddr = null;
            srcAddr = null;
            if (inputStr == null) return 0;
            if (inputStr.Length == 0) return 0;

            if (inputStr.Contains(BinaryDataConverter.LABEL_ADDRESS_MARKER) == true)
            {
                int foo = 1;
                int bar = foo;
            }

            for (int i = 0; i < inputStr.Length; i++)
            {
                // are we on the start of a label
                if (inputStr[i] == BinaryDataConverter.LABEL_ADDRESS_MARKER_AS_CHAR)
                {
                    inAddrNow = true;
                    continue;
                }
                // are we in a label and have we reached the end of the label?
                else if( (inAddrNow == true) && ((inputStr[i] == ' ') || (inputStr[i] == '\r') || (inputStr[i] == '\n')))
                {
                    inAddrNow = false;
                    // should we store it in the first one?
                    if (destAddr == null)
                    {
                        // yes, store it here
                        destAddr = sbAddr.ToString();
                    }
                    else
                    {
                        // no, put it in the second one
                        srcAddr = sbAddr.ToString();
                    }
                    // always rebuild this
                    sbAddr = new StringBuilder();
                    continue;
                }
                // are we in a label (and implied by above) not at the end
                else if (inAddrNow == true)
                {
                    sbAddr.Append(inputStr[i]);
                    continue;
                }
                else
                {
                    // not in a label - just ignore
                    continue;
                }
            } // bottom of for (int i = 0; i < inputStr.Length; i++)

            // send an appropriate return signal
            if (destAddr != null) return 1;
            if (srcAddr != null) return 2;
            return 0;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the clock frequency in Mhz out of the init area.
        /// </summary>
        /// <returns>string with the clockFrequency in Mhz. This is returned in decimal</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetClockFreq()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertFourLittleEndianBytesToInteger(binaryFileInMemory[0], binaryFileInMemory[1], binaryFileInMemory[2], binaryFileInMemory[3]);
            return retInt.ToString();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the program base out of the init area.
        /// </summary>
        /// <returns>string with the program base. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetProgramBaseAsStr()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[6], binaryFileInMemory[7]);
            string retStr = System.Convert.ToString(retInt, 16);
            return BinaryDataConverter.PadWithLeadingZeros(retStr, 4).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the program base out of the init area.
        /// </summary>
        /// <returns>string with the program base. This is returned in hex</returns>
        /// <history>
        ///    11 Apr 09  Cynic - originally written
        /// </history>
        public int GetProgramBaseAsInt()
        {
            if (binaryFileInMemory == null) return -1;
            if (binaryFileInMemory.Length < 16) return -1;
            return BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[6], binaryFileInMemory[7]);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the variable base out of the init area.
        /// </summary>
        /// <returns>string with the variable base. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetVariableBaseAsStr()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[8], binaryFileInMemory[9]);
            string retStr = System.Convert.ToString(retInt, 16);
            return BinaryDataConverter.PadWithLeadingZeros(retStr, 4).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the variable base out of the init area.
        /// </summary>
        /// <returns>string with the variable base. This is returned in hex</returns>
        /// <history>
        ///    11 Apr 09  Cynic - originally written
        /// </history>
        public int GetVariableBaseAsInt()
        {
            if (binaryFileInMemory == null) return -1;
            if (binaryFileInMemory.Length < 16) return -1;
            return BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[8], binaryFileInMemory[9]);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the stack base out of the init area.
        /// </summary>
        /// <returns>string with the stack base. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetStackBaseAsStr()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[10], binaryFileInMemory[11]);
            string retStr = System.Convert.ToString(retInt, 16);
            return BinaryDataConverter.PadWithLeadingZeros(retStr, 4).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the stack base out of the init area.
        /// </summary>
        /// <returns>string with the stack base. This is returned in hex</returns>
        /// <history>
        ///    11 Apr 09  Cynic - originally written
        /// </history>
        public int GetStackBaseAsInt()
        {
            if (binaryFileInMemory == null) return -1;
            if (binaryFileInMemory.Length < 16) return -1;
            return BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[10], binaryFileInMemory[11]);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the InitialProgramCounter out of the init area.
        /// </summary>
        /// <returns>string with the stack base. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetInitialProgramCounterAsStr()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[12], binaryFileInMemory[13]);
            string retStr = System.Convert.ToString(retInt, 16);
            return BinaryDataConverter.PadWithLeadingZeros(retStr, 4).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the InitialProgramCounter out of the init area.
        /// </summary>
        /// <returns>string with the stack base. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public int GetInitialProgramCounterAsInt()
        {
            if (binaryFileInMemory == null) return -1;
            if (binaryFileInMemory.Length < 16) return -1;
            return BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[12], binaryFileInMemory[13]);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the InitialStackCounter out of the init area.
        /// </summary>
        /// <returns>string with the stack base. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetInitialStackCounterAsStr()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[14], binaryFileInMemory[15]);
            string retStr = System.Convert.ToString(retInt, 16);
            return BinaryDataConverter.PadWithLeadingZeros(retStr, 4).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the InitialStackCounter out of the init area.
        /// </summary>
        /// <returns>string with the stack base. This is returned in hex</returns>
        /// <history>
        ///    11 Apr 09  Cynic - originally written
        /// </history>
        public int GetInitialStackCounterAsInt()
        {
            if (binaryFileInMemory == null) return -1;
            if (binaryFileInMemory.Length < 16) return -1;
            return BinaryDataConverter.ConvertTwoLittleEndianBytesToInteger(binaryFileInMemory[14], binaryFileInMemory[15]);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the GetXTALMode out of the init area.
        /// </summary>
        /// <returns>string with the XTALMode. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetXTALMode()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertOneLittleEndianByteToInteger(binaryFileInMemory[4]);
            string retStr = System.Convert.ToString(retInt, 16);
            return BinaryDataConverter.PadWithLeadingZeros(retStr, 2).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the Checksum out of the init area.
        /// </summary>
        /// <returns>string with the Checksum. This is returned in hex</returns>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public string GetChecksum()
        {
            if (binaryFileInMemory == null) return "";
            if (binaryFileInMemory.Length < 16) return "";
            int retInt = BinaryDataConverter.ConvertOneLittleEndianByteToInteger(binaryFileInMemory[5]);
            string retStr = System.Convert.ToString(retInt, 16);
            return BinaryDataConverter.PadWithLeadingZeros(retStr, 2).ToUpper();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Reset the class
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - originally written
        /// </history>
        public void Reset()
        {
            binaryFileName=null;
            binaryFileInMemory=null;
        }
    }
}
