using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;

/// +------------------------------------------------------------------------------------------------------------------------------+
///                                                    TERMS OF USE: MIT License                                                  
/// +------------------------------------------------------------------------------------------------------------------------------
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    
/// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
/// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
/// is furnished to do so, subject to the following conditions:                                                                   
///                                                                                                                               
/// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
///                                                                                                                               
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
/// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
/// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
/// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
/// +------------------------------------------------------------------------------------------------------------------------------+

namespace PropScriptBitInspector
{
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// <summary>
    /// A class to provide the basic form container for the PropScriptBitInspector
    /// </summary>
    /// <history>
    ///    09 Apr 09  Cynic - Started
    /// </history>

    public partial class frmMain : Form
    {
#if DEBUG
        private string DEBUG_DIR = @"C:\PropScriptBinCode";
#else
        private string DEBUG_DIR = @".";
#endif 

        private const string DEFAULT_DISPLAY_INSTRUCTION = "5C 7C 00 02";

        private BinaryFileFormatter binaryFileFormatterObj = null;

        // tracks the current line in the binary file
        private int lastCurrentLineStart = -1;
        private int lastCurrentLineEnd = -1;
        private bool currentlyAdjustingSelection = false;

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Constructor for the main form
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        public frmMain()
        {
            InitializeComponent();

            // set up
            this.textBoxInputValue.Text = DEFAULT_DISPLAY_INSTRUCTION;
            SyncDisplayOptionsToBinaryFormatter();
            SyncDESTFieldLabelOptionsToReality();
        }

        // ####################################################################
        // ##### Code to manage the Instruction Tab ###########################
        // ####################################################################

        #region Instruction Tab Handling Code

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles state changes on the InputValueType checkboxes
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private void radioButtonInputValueType_CheckedChanged(object sender, EventArgs e)
        {
            ConvertInputValue();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Converts the input value based on the state of the radio buttons
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private void ConvertInputValue()
        {
            const string NO_CONVERSION = "No Conversion";
            string workingString;
            int workingInt=65;
            string hexStr;
            string binaryStr;
            string binaryLEStr;
            string hexLEStr;

            textBoxStdBinary.Text = "";
            textBoxLEBinary.Text = "";
            textBoxStdHex.Text = "";
            textBoxLEHex.Text = "";
            textBoxErrorMessage.Text = "";
            ctlDisplayINSTRPanel.Reset();

            if(textBoxInputValue.Text==null) goto FAIL_OUT;
            workingString=textBoxInputValue.Text.Trim();

            // remove all spaces
            if(workingString.Length==0) goto FAIL_OUT;
            workingString = workingString.Replace(" ", "");

            // what does the user think the input format is?
            if(radioButtonIsStdBinary.Checked==true)
            {
                // the input is standard binary
                try
                {
                    workingInt = Convert.ToInt32(workingString, 2);
                }
                catch (Exception ex)
                {
                    textBoxErrorMessage.Text = ex.Message;
                    goto FAIL_OUT;
                }
            }
            else if (radioButtonIsLEBinary.Checked == true)
            {
                // the input is little endian binary
                try
                {
                    binaryStr = BinaryDataConverter.ConvertLittleEndianBinaryStrToBinaryStr(workingString);
                    workingInt = Convert.ToInt32(binaryStr, 2);
                }
                catch (Exception ex)
                {
                    textBoxErrorMessage.Text = ex.Message;
                    goto FAIL_OUT;
                }
            }
            else if (radioButtonIsStdHex.Checked == true)
            {
                // the input is standard hex
                try
                {
                    workingInt = Convert.ToInt32(workingString, 16);
                }
                catch (Exception ex)
                {
                    textBoxErrorMessage.Text = ex.Message;
                    goto FAIL_OUT;
                }
            }
            else if(radioButtonIsLEHex.Checked==true)
            {
                // the input is little endian hex
                try
                {
                    hexStr = BinaryDataConverter.ConvertLittleEndianHexStrToHexStr(workingString);
                    workingInt = Convert.ToInt32(hexStr, 16);
                }
                catch (Exception ex)
                {
                    textBoxErrorMessage.Text = ex.Message;
                    goto FAIL_OUT;
                }
            }
            else
            {
                // should never happen
                goto FAIL_OUT;
            }

            // display the binary strings
            binaryStr = System.Convert.ToString(workingInt, 2);
            binaryStr = BinaryDataConverter.PadWithLeadingZeros(binaryStr, 32);
            binaryLEStr = BinaryDataConverter.ConvertBinaryStrToLittleEndian(binaryStr);
            textBoxStdBinary.Text = BinaryDataConverter.InsertSpaceSeparators(binaryStr, 8);
            textBoxLEBinary.Text = BinaryDataConverter.InsertSpaceSeparators(binaryLEStr, 8);

            // now display the Prop ASM Code Split Out
            byte[] bytes = System.BitConverter.GetBytes(workingInt);
            ctlDisplayINSTRPanel.SetDisplay(0, bytes[0], bytes[1], bytes[2], bytes[3]);            

            // now the hex boxes
            hexStr = System.Convert.ToString(workingInt, 16);
            hexStr = BinaryDataConverter.PadWithLeadingZeros(hexStr, 8).ToUpper();
            hexLEStr = BinaryDataConverter.ConvertHexStrToLittleEndian(hexStr).ToUpper();
            textBoxStdHex.Text = BinaryDataConverter.InsertSpaceSeparators(hexStr, 2);
            textBoxLEHex.Text = BinaryDataConverter.InsertSpaceSeparators(hexLEStr, 2);

            return;
            FAIL_OUT:
                textBoxStdBinary.Text = NO_CONVERSION;
                textBoxLEBinary.Text = NO_CONVERSION;
                textBoxStdHex.Text = NO_CONVERSION;
                textBoxLEHex.Text = NO_CONVERSION;
            }

            /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
            /// <summary>
            /// Handles text changed operations
            /// </summary>
            /// <history>
            ///    09 Apr 09  Cynic - Started
            /// </history>
            private void textBoxInputValue_TextChanged(object sender, EventArgs e)
            {
                ConvertInputValue();
            }

            /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
            /// <summary>
            /// Handles checked changed operations
            /// </summary>
            /// <history>
            ///    09 Apr 09  Cynic - Started
            /// </history>
            private void radioButtonIsStdBinary_CheckedChanged(object sender, EventArgs e)
            {
                ConvertInputValue();
            }

            /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
            /// <summary>
            /// Handles checked changed operations
            /// </summary>
            /// <history>
            ///    09 Apr 09  Cynic - Started
            /// </history>
            private void radioButtonIsLEBinary_CheckedChanged(object sender, EventArgs e)
            {
                ConvertInputValue();
            }

            /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
            /// <summary>
            /// Handles checked changed operations
            /// </summary>
            /// <history>
            ///    09 Apr 09  Cynic - Started
            /// </history>
            private void radioButtonIsStdHex_CheckedChanged(object sender, EventArgs e)
            {
                ConvertInputValue();
            }

            /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
            /// <summary>
            /// Handles checked changed operations
            /// </summary>
            /// <history>
            ///    09 Apr 09  Cynic - Started
            /// </history>
            private void radioButtonIsLEHex_CheckedChanged(object sender, EventArgs e)
            {
                ConvertInputValue();
            }

            /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
            /// <summary>
            /// Handles clicks on the convert button
            /// </summary>
            /// <history>
            ///    09 Apr 09  Cynic - Started
            /// </history>
            private void buttonConvert_Click(object sender, EventArgs e)
            {
                ConvertInputValue();
            }

        #endregion

        // ####################################################################
        // ##### Code to manage the Binary Display Tab#########################
        // ####################################################################

        #region Binary File Tab Handling Code

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the name of a binary or eeprom file to open
        /// </summary>
        /// <param name="lastFileOrDir">we use this to set the default directory</param>
        /// <returns>file name or null</returns>
        /// <history>
        ///    10 Apr 09  Cynic - Originally Written
        /// </history>
        private string GetBinaryFileName(string lastFileOrDir)
        {
            OpenFileDialog fileOpenForm = null;
            try
            {
                fileOpenForm = new OpenFileDialog();
                if (lastFileOrDir != null)
                {
                    if (Directory.Exists(lastFileOrDir) == true) fileOpenForm.InitialDirectory = lastFileOrDir;
                    else fileOpenForm.InitialDirectory = Path.GetDirectoryName(lastFileOrDir);
                }
                fileOpenForm.DefaultExt = ".binary";
                fileOpenForm.Filter = "BINARY Files(*.binary;*.eeprom)|*.binary;*.eeprom|All files (*.*)|*.*";
                fileOpenForm.Title = "Open .BINARY or .EEPROM File...";
                if (fileOpenForm.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    return fileOpenForm.FileName;
                }
                else
                {
                    return null;
                }
            }
            catch (System.Exception ex)
            {
                string errStr = "Error opening file: " + ex.Message;
                textBoxErrorMessage.Text = errStr;
                MessageBox.Show(errStr);
                return null;

            }
            finally
            {
                if(fileOpenForm!=null) fileOpenForm.Dispose();
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Opens up a binary or eeprom file and formats it for the display
        /// </summary>
        /// <returns>file name or null</returns>
        /// <history>
        ///    10 Apr 09  Cynic - Originally Written
        /// </history>
        private void DisplayFileFromDisk(string fileToOpen)
        {
            string errStr = null;
            int retInt;

            ClearBinaryFileDisplay();
            if (binaryFileFormatterObj != null) binaryFileFormatterObj.Reset();

            if (fileToOpen == null)
            {
                errStr = "fileToOpen == null";
                textBoxErrorMessage.Text = errStr;
                MessageBox.Show(errStr);
                return;
            }

            if (File.Exists(fileToOpen) == false)
            {
                errStr = "File does not exist: " + fileToOpen;
                textBoxErrorMessage.Text = errStr;
                MessageBox.Show(errStr);
                return;
            }

            // reset this
            binaryFileFormatterObj = new BinaryFileFormatter();
            retInt=binaryFileFormatterObj.ReadFileIntoMemory(fileToOpen, ref errStr);
            if (retInt !=0 )
            {
                textBoxErrorMessage.Text = "Binary File Read Failed: ErrCode="+retInt.ToString()+" err="+errStr;
                MessageBox.Show(errStr);
                return;
            }
            // make sure the new binary formatter knows about the options
            SyncBinaryFormatterOptionsToDisplay();
            // display it on the screen
            RefreshBinaryFileDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Refreshes the binary file display
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Originally Written
        /// </history>
        private void RefreshBinaryFileDisplay()
        {
            lastCurrentLineStart = -1;
            lastCurrentLineEnd = -1;
            currentlyAdjustingSelection = false;

            if (binaryFileFormatterObj == null)
            {
                ClearBinaryFileDisplay();
                return;
            }
            if ((binaryFileFormatterObj.BinaryFileName == null) || (binaryFileFormatterObj.BinaryFileName.Length==0))
            {
                ClearBinaryFileDisplay();
                return;
            }

            richTextBoxBinaryDisplay.Text = binaryFileFormatterObj.GetDisplayText();
            textBoxClockFreq.Text = binaryFileFormatterObj.GetClockFreq();
            textBoxXTALMode.Text = binaryFileFormatterObj.GetXTALMode();
            textBoxCheckSum.Text = binaryFileFormatterObj.GetChecksum();
            textBoxProgramBase.Text = binaryFileFormatterObj.GetProgramBaseAsStr();
            textBoxVariableBase.Text = binaryFileFormatterObj.GetVariableBaseAsStr();
            textBoxStackBase.Text = binaryFileFormatterObj.GetStackBaseAsStr();
            textBoxInitialProgramCounter.Text = binaryFileFormatterObj.GetInitialProgramCounterAsStr();
            textBoxInitialStackCounter.Text = binaryFileFormatterObj.GetInitialStackCounterAsStr();

            // need to display the filename
            textBoxCurrentOpenFile.Text = binaryFileFormatterObj.BinaryFileName;

            ClearAllBoldFontFromBinaryDisplay();

        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Clear the Binary Display file
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Originally Written
        /// </history>
        private void ClearBinaryFileDisplay()
        {
            lastCurrentLineStart = -1;
            lastCurrentLineEnd = -1;
            currentlyAdjustingSelection = false;

            richTextBoxBinaryDisplay.Clear();
            richTextBoxBinaryDisplay.Text = "";
            textBoxClockFreq.Text = "";
            textBoxXTALMode.Text = "";
            textBoxCheckSum.Text = "";
            textBoxProgramBase.Text = "";
            textBoxVariableBase.Text = "";
            textBoxStackBase.Text = "";
            textBoxInitialProgramCounter.Text = "";
            textBoxInitialStackCounter.Text = "";
            textBoxCurrentOpenFile.Text = "";
            ctlDisplayINSTRFilePanel.Reset();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Writes the binary formatter options back to the display and refreshes
        /// the display
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private void SyncBinaryFormatterOptionsToDisplay()
        {
            if (binaryFileFormatterObj == null) return;
            binaryFileFormatterObj.WantBinaryValues = checkBoxShowBinaryValues.Checked;
            binaryFileFormatterObj.WantPasmCodes = checkBoxShowPASMCode.Checked;
            binaryFileFormatterObj.WantProgramCounter = checkBoxShowProgramCounter.Checked;
            binaryFileFormatterObj.RemoveLittleEndian=radioButtonUndoLittleEndian.Checked;
            binaryFileFormatterObj.DisplayLabelsAsText = this.radioButtonLabelsAsText.Checked;
            binaryFileFormatterObj.DisplayLabelHexAsAbsolute = this.checkBoxLabelsAsAbsoluteHexAddr.Checked;
            binaryFileFormatterObj.InterpretEverythingAsPASM = this.checkBoxInterpretEverythingAsPASM.Checked;

            RefreshBinaryFileDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Writes the binary formatter options back to the display and refreshes
        /// the display
        /// </summary>
        /// <history>
        ///    09 Apr 09  Cynic - Started
        /// </history>
        private void SyncDisplayOptionsToBinaryFormatter()
        {
            if (binaryFileFormatterObj == null)
            {
                checkBoxShowBinaryValues.Checked = BinaryFileFormatter.DEFAULT_WANT_BINARY_VALUES;
                checkBoxShowPASMCode.Checked = BinaryFileFormatter.DEFAULT_WANT_PASM_CODES;
                checkBoxShowProgramCounter.Checked = BinaryFileFormatter.DEFAULT_WANT_PROGRAM_COUNTER;
                radioButtonUndoLittleEndian.Checked = BinaryFileFormatter.DEFAULT_REMOVE_LITTLE_ENDIAN;
                radioButtonDisplayBytesAsIs.Checked = !BinaryFileFormatter.DEFAULT_REMOVE_LITTLE_ENDIAN;
                radioButtonLabelsAsText.Checked = BinaryFileFormatter.DEFAULT_DISPLAY_LABELS_AS_TEXT;
                checkBoxLabelsAsAbsoluteHexAddr.Checked = BinaryFileFormatter.DEFAULT_DISPLAY_LABEL_HEX_AS_ABSOLUTE;
                checkBoxInterpretEverythingAsPASM.Checked = BinaryFileFormatter.DEFAULT_INTERPRET_EVERYTHING_AS_PASM;
                return;
            }
            checkBoxShowBinaryValues.Checked=binaryFileFormatterObj.WantBinaryValues;
            checkBoxShowPASMCode.Checked = binaryFileFormatterObj.WantPasmCodes;
            checkBoxShowProgramCounter.Checked = binaryFileFormatterObj.WantProgramCounter;
            radioButtonUndoLittleEndian.Checked = binaryFileFormatterObj.RemoveLittleEndian;
            radioButtonLabelsAsText.Checked = binaryFileFormatterObj.DisplayLabelsAsText;
            checkBoxLabelsAsAbsoluteHexAddr.Checked = binaryFileFormatterObj.DisplayLabelHexAsAbsolute;
            checkBoxInterpretEverythingAsPASM.Checked = binaryFileFormatterObj.InterpretEverythingAsPASM;
            RefreshBinaryFileDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Displays the broken out components for the current line on the display
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        public void DisplayComponentsForPASMLine(int line)
        {
            int offset=(line*4);
            byte byte0 = 0;
            byte byte1 = 0;
            byte byte2 = 0;
            byte byte3 = 0;

            if (binaryFileFormatterObj == null)
            {
                ctlDisplayINSTRFilePanel.Reset();
                return;
            }
            // get the bytes
            binaryFileFormatterObj.GetBytesAtOffset(offset, ref byte0, ref byte1, ref byte2, ref byte3);
            // display the line
            ctlDisplayINSTRFilePanel.SetDisplay(offset, byte0, byte1, byte2, byte3);
            return;

        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a click on the button which opens a binary or eeprom file
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        private void buttonOpenFile_Click(object sender, EventArgs e)
        {
            string defaultDir=null;

            // set the default directory using the last known filename
            if (binaryFileFormatterObj != null) defaultDir = binaryFileFormatterObj.BinaryFileName;
            if (defaultDir == null) defaultDir = DEBUG_DIR;
            // get the file
            string fileToOpen = GetBinaryFileName(defaultDir);
            // error message will already have been displayed
            if (fileToOpen == null) return;
            // display the file
            DisplayFileFromDisk(fileToOpen);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a click on the checkbox which enables program counter columns to display
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        private void checkBoxShowProgramCounter_CheckedChanged(object sender, EventArgs e)
        {
            SyncBinaryFormatterOptionsToDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a click on the checkbox which enables forces us to interpret everything as PASM
        /// </summary>
        /// <history>
        ///    26 Apr 09  Cynic - Started
        /// </history>
        private void checkBoxInterpretEverythingAsPASM_CheckedChanged(object sender, EventArgs e)
        {
            SyncBinaryFormatterOptionsToDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a click on the checkbox which enables Binary Values columns to display
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        private void checkBoxShowBinaryValues_CheckedChanged(object sender, EventArgs e)
        {
            SyncBinaryFormatterOptionsToDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a click on the checkbox which enables PASM Code columns to display
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        private void checkBoxShowPASMCode_CheckedChanged(object sender, EventArgs e)
        {
            SyncShowLabelRadioButtonsToReality();
            SyncBinaryFormatterOptionsToDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a click on the radio buttons which control the display order
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        private void radioButtonDisplayOrder_CheckedChanged(object sender, EventArgs e)
        {
            SyncBinaryFormatterOptionsToDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a selection changed value for the display control
        /// </summary>
        /// <history>
        ///    10 Apr 09  Cynic - Started
        /// </history>
        private void richTextBoxBinaryDisplay_SelectionChanged(object sender, EventArgs e)
        {
            // find out the line is on 
            int pos = this.richTextBoxBinaryDisplay.SelectionStart;
            int line = this.richTextBoxBinaryDisplay.GetLineFromCharIndex(pos);
            DisplayComponentsForPASMLine(line);
            BoldFontCurrentLineOfBinaryFile();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a checked changed event on the "How do we display DEST labels"
        /// radio button group
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        private void radioButtonLabelDisplay_CheckedChanged(object sender, EventArgs e)
        {
            SyncDESTFieldLabelOptionsToReality();
            SyncBinaryFormatterOptionsToDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// enables or disables the checkBoxLabelsAsAbsoluteHexAddr option
        /// if the appropriate radio button is not checked
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        private void SyncDESTFieldLabelOptionsToReality()
        {
            if ((radioButtonLabelsAsHex.Checked == true) && (radioButtonLabelsAsHex.Enabled == true))
            {
                checkBoxLabelsAsAbsoluteHexAddr.Enabled = true;
            }
            else
            {
                checkBoxLabelsAsAbsoluteHexAddr.Enabled = false;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// enables or disables the radioButtonLabel options
        /// if the appropriate radio button is not checked
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        private void SyncShowLabelRadioButtonsToReality()
        {
            if (checkBoxShowPASMCode.Checked == true)
            {
                this.radioButtonLabelsAsText.Enabled = true;
                this.radioButtonLabelsAsHex.Enabled = true;
            }
            else
            {
                this.radioButtonLabelsAsText.Enabled = false;
                this.radioButtonLabelsAsHex.Enabled = false;
            }
            SyncDESTFieldLabelOptionsToReality();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a checked changed event on the "How do we display absolue addrs"
        /// check box
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        private void checkBoxLabelsAsAbsoluteHexAddr_CheckedChanged(object sender, EventArgs e)
        {
            SyncBinaryFormatterOptionsToDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Marks the current liine (the one with the carat) in the BinaryFile
        /// as bold, also unbolds the previous one
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        private void BoldFontCurrentLineOfBinaryFile()
        {
            if (currentlyAdjustingSelection == true) return;

            try
            {
                currentlyAdjustingSelection = true;
                int currentSelectionStart = richTextBoxBinaryDisplay.SelectionStart;
                int currentSelectionLength = richTextBoxBinaryDisplay.SelectionLength;

                // get the current line position of the cursor
                int tmpCurrentLineStart = richTextBoxBinaryDisplay.GetFirstCharIndexOfCurrentLine();
                int currentLine = richTextBoxBinaryDisplay.GetLineFromCharIndex(tmpCurrentLineStart);
                int tmpCurrentLineEnd = richTextBoxBinaryDisplay.GetFirstCharIndexFromLine(currentLine + 1);

                // turn off the bolding on the previoius line
                if ((lastCurrentLineStart >= 0) && (lastCurrentLineEnd >= 0) && (lastCurrentLineStart <= lastCurrentLineEnd))
                {
                    // remove the previous selection
                    richTextBoxBinaryDisplay.Select(lastCurrentLineStart, lastCurrentLineEnd - lastCurrentLineStart);
                    richTextBoxBinaryDisplay.SelectionFont = new Font(richTextBoxBinaryDisplay.SelectionFont, FontStyle.Regular);
                    lastCurrentLineStart = -1;
                    lastCurrentLineEnd = -1;
                }

                // now make the current line bold
                if ((tmpCurrentLineStart >= 0) && (tmpCurrentLineEnd >= 0) && (tmpCurrentLineStart <= tmpCurrentLineEnd))
                {
                    richTextBoxBinaryDisplay.Select(tmpCurrentLineStart, tmpCurrentLineEnd - tmpCurrentLineStart);
                    richTextBoxBinaryDisplay.SelectionFont = new Font(richTextBoxBinaryDisplay.SelectionFont, FontStyle.Bold);
                }

                // remember the previous settings
                lastCurrentLineStart = tmpCurrentLineStart;
                lastCurrentLineEnd = tmpCurrentLineEnd;

                if ((currentSelectionStart >= 0) && (currentSelectionLength >= 0))
                {
                    richTextBoxBinaryDisplay.Select(currentSelectionStart, currentSelectionLength);
                }
            }
            finally
            {
                currentlyAdjustingSelection = false;
            }

        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Clears all bold font attributes from all characters in the binary file display
        /// </summary>
        /// <history>
        ///    18 Apr 09  Cynic - Started
        /// </history>
        private void ClearAllBoldFontFromBinaryDisplay()
        {
            if (currentlyAdjustingSelection == true) return;
            try
            {
                currentlyAdjustingSelection = true;

                lastCurrentLineStart = -1;
                lastCurrentLineEnd = -1;
                currentlyAdjustingSelection = false;

                richTextBoxBinaryDisplay.Select(0, richTextBoxBinaryDisplay.Text.Length);
                richTextBoxBinaryDisplay.SelectionFont = new Font(richTextBoxBinaryDisplay.SelectionFont, FontStyle.Regular);
                richTextBoxBinaryDisplay.Select(0, 0);
            }
            catch
            {
            }
            finally
            {
                currentlyAdjustingSelection = false;
            }
        }
        #endregion

        private void button1_Click(object sender, EventArgs e)
        {
            ClearAllBoldFontFromBinaryDisplay();
            //richTextBoxBinaryDisplay.Select(0, richTextBoxBinaryDisplay.Text.Length);
           // richTextBoxBinaryDisplay.SelectionFont = new Font(richTextBoxBinaryDisplay.SelectionFont, FontStyle.Regular);
            //richTextBoxBinaryDisplay.SelectionFont = new Font("Tahoma", 12, FontStyle.Italic);
        }

   }
}