using System;
using System.Collections.Generic;
using System.Text;

/// +------------------------------------------------------------------------------------------------------------------------------+
///                                                    TERMS OF USE: MIT License                                                  
/// +------------------------------------------------------------------------------------------------------------------------------
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    
/// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
/// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
/// is furnished to do so, subject to the following conditions:                                                                   
///                                                                                                                               
/// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
///                                                                                                                               
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
/// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
/// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
/// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
/// +------------------------------------------------------------------------------------------------------------------------------+

namespace SRV1CSharpConsole
{
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// <summary>
    /// A class to provide a standard mechanism for the transport of responses
    /// from the SRV1.
    /// 
    /// </summary>
    /// <history>
    ///    24 Nov 08  Cynic - Originally written
    /// </history>
    public class SRV1Response
    {
        private int errorCode = 0;
        private SRVResponseTypeEnum responseType = SRVResponseTypeEnum.ERROR;

        // note, the length of this value is always stores the total length of the 
        // data with no spare bytes. So you can use responseData.Length to see
        // what amount of data you are dealing with.
        private byte[] responseData = null;

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Constructor, assumes responseType = SRVResponseTypeEnum.ERROR;
        /// </summary>
        /// <param name="errorCodeIn">The error code</param>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        /// </history>
        public SRV1Response(int errorCodeIn)
        {
            errorCode=errorCodeIn;
            responseType = SRVResponseTypeEnum.ERROR;
            responseData = null;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Constructor, assumes responseType = SRVResponseTypeEnum.ERROR;
        /// </summary>
        /// <param name="errorCodeIn">The error code</param>
        /// <param name="responseTypeIn">The response type</param>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        /// </history>
        public SRV1Response(int errorCodeIn, SRVResponseTypeEnum responseTypeIn)
        {
            errorCode=errorCodeIn;
            responseType = responseTypeIn;
            responseData = null;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets the error code value. This is a user defined value meaninful
        /// only to the creator of this class
        /// </summary>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        /// </history>
        public int ErrorCode
        {
            get
            {
                return errorCode;
            }
            set
            {
                errorCode = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets the responseType value. This is a broad classification which
        /// describes how the code using this class should interpret the response data
        /// </summary>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        /// </history>
        public SRVResponseTypeEnum ResponseType
        {
            get
            {
                return responseType;
            }
            set
            {
                responseType = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Set the binary response data.
        /// </summary>
        /// <param name="dataLen">the length of the data</param>
        /// <param name="dataStart">the start position in the data array</param>
        /// <param name="responseDataIn">a byte array of the data</param>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        /// </history>
        public void SetResponseData(byte [] responseDataIn, int dataStart, int dataLen )
        {
            // init
            responseData=null;
            // got anything
            if(responseDataIn==null) return;
            if(dataStart<0) return;
            if(dataLen<=0) return;
            // we must not read off the end
            if(responseDataIn.Length<dataStart+dataLen) return;

            // create a new response data array
            responseData=new byte[dataLen];
            // copy it in as fast as possible
            Buffer.BlockCopy(responseDataIn, dataStart, responseData, 0, dataLen); 
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the responseData value as a string. Not appropriate for all
        /// responseTypes. It is up to the caller to make sure this is appropriate
        /// </summary>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        /// </history>
        public string ResponseDataAsString
        {
            get
            {
                if (responseData == null) return "";
                else return System.Text.Encoding.ASCII.GetString(responseData, 0, responseData.Length); ;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the responseData value as a byte[]. 
        /// </summary>
        /// <history>
        ///    26 Nov 08  Cynic - Originally written
        /// </history>
        public byte[] ResponseDataAsByteArray
        {
            get
            {
                return responseData;
            }
        }
    }
}

/// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
/// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
/// <summary>
/// A class to define the various standard response types of the SRV1
/// </summary>
/// <history>
///    24 Nov 08  Cynic - Originally written
/// </history>
public enum SRVResponseTypeEnum
{
    ERROR = 0,
    TIMED_OUT = 1,
    STD_1CHAR_TEXT = 2,
    STD_1CHAR_AND_BINARY = 3,
    STD_IMAGE = 4,
    STD_TEXTONLY = 5,
    ZDUMP = 6,
    UNKNOWN = 7
}