using System;
using System.Collections.Generic;
using System.Text;
using System.Net;
using System.Net.Sockets;

/// +------------------------------------------------------------------------------------------------------------------------------+
///                                                    TERMS OF USE: MIT License                                                  
/// +------------------------------------------------------------------------------------------------------------------------------
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    
/// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
/// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
/// is furnished to do so, subject to the following conditions:                                                                   
///                                                                                                                               
/// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
///                                                                                                                               
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
/// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
/// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
/// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
/// +------------------------------------------------------------------------------------------------------------------------------+

namespace SRV1CSharpConsole
{
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// <summary>
    /// A class to buffered reading of a TCPStream with multibyte peek ahead
    /// (something NetworkStream really ought to have IMHO)
    /// </summary>
    /// <history>
    ///    02 Dec 08  Cynic - Originally written
    /// </history>
    class TCPBufferedStream
    {
        private NetworkStream netStream = null;
        private object bufferLock = new object();

        // our buffer management items
        // we use a 10mb buffer as default
        private const int DEFAULT_BUFFER_SIZE = (1024 * 1024 * 10);
        private byte[] netBuffer = null;
        private int bufferSize = DEFAULT_BUFFER_SIZE;
        private int bufferNextFreeSlot = 0;

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Constructor
        /// </summary>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public TCPBufferedStream()
        {
            netBuffer = new byte[DEFAULT_BUFFER_SIZE];
            bufferSize = DEFAULT_BUFFER_SIZE;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="bufferSize">the size of the data we are prepared to buffer</param>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public TCPBufferedStream(int bufferSizeIn)
        {
            // create our buffer 
            if (bufferSize <= 0)
            {
                netBuffer = new byte[bufferSizeIn];
                bufferSize = bufferSizeIn;
            }
            else
            {
                netBuffer = new byte[DEFAULT_BUFFER_SIZE];
                bufferSize = DEFAULT_BUFFER_SIZE;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets whether there is data available in the buffer
        /// </summary>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public bool DataAvailable
        {
            get
            {
                if (bufferNextFreeSlot != 0) return true;
                return false;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets/Sets the current network stream
        /// </summary>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public NetworkStream NetStream
        {
            get
            {
                return netStream;
            }
            set
            {
                netStream = value;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Resets the buffer so that we think we have nothing in it
        /// </summary>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public void ResetBuffer()
        {
            bufferNextFreeSlot = 0;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Reads all data from the network stream and stores it in the internal buffer
        /// This may have an arbitrary number of complete or partially complete packets in it
        /// </summary>
        /// <returns>the number of bytes read or -ve for fail</returns>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public int ConsumeStreamData()
        {
            Int32 bytesRead = 0;

            lock (bufferLock)
            {
                if (netBuffer == null) return -97;
                if (netStream == null) return -99;
                if (bufferNextFreeSlot >= bufferSize) return -98;
                // any data there?
                if (netStream.DataAvailable == false)
                {
                    return LengthOfBufferData;
                }

                // Read the netStream bytes here
                bytesRead = netStream.Read(netBuffer, bufferNextFreeSlot, (bufferSize - bufferNextFreeSlot));
                // add this on here, so we tack onto the end next time
                bufferNextFreeSlot += bytesRead;
                return bytesRead;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Returns the length of the data stored in the buffer. This is always
        /// the bufferNextFreeSlot
        /// </summary>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public int LengthOfBufferData
        {
            get
            {
                lock(bufferLock)
                {
                    return this.bufferNextFreeSlot;
                }
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the requested amount of data bytes from the buffer
        /// </summary>
        /// <param name="numBytes">Get the requested amount of data. Return null if
        /// that much data is not available (or error)</param>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public byte[] GetDataBytes(int numBytes)
        {
            lock (bufferLock)
            {
                // some sanity checks
                if (netBuffer == null) return null;
                if (netStream == null) return null;
                if (numBytes <= 0) return null;
                if (numBytes > LengthOfBufferData) return null;

                // build the return buffer
                byte[] retBuffer = new byte[numBytes];
                // copy it in as fast as possible
                Buffer.BlockCopy(netBuffer, 0, retBuffer, 0, numBytes);
                // now remove the data we just returned from the netBuffer
                Buffer.BlockCopy(netBuffer, numBytes, netBuffer, 0, (LengthOfBufferData - numBytes));
                // reset this to the new 
                bufferNextFreeSlot = (LengthOfBufferData - numBytes);
                return retBuffer;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Peeks into the returning data and finds the number of bytes to consume
        /// in order to make that byte the next available byte in the buffer
        /// </summary>
        /// <param name="byteToLookFor">The byte value we are looking for</param>
        /// <returns>the number of bytes to consume, or -1 for fail</returns>
        /// <history>
        ///    03 Dec 08  Cynic - Originally written
        /// </history>
        public int LocateByte(byte byteToLookFor)
        {
            lock (bufferLock)
            {
                // some sanity checks
                if (netBuffer == null) return -1;
                if (netStream == null) return -1;
                if (LengthOfBufferData <= 0) return -1;
                for (int i = 0; i < LengthOfBufferData; i++)
                {
                    if (netBuffer[i] == byteToLookFor) return i;
                }
                // if we get here the byteToLookFor was not found
                return -1;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the requested amount of data bytes from the buffer, but only peeks
        /// into it and does not remove the returned data
        /// </summary>
        /// <param name="numBytes">Get the requested amount of data. Return null if
        /// that much data is not available (or error)</param>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public byte[] PeekDataBytes(int numBytes)
        {
            lock (bufferLock)
            {
                // some sanity checks
                if (netBuffer == null) return null;
                if (netStream == null) return null;
                if (numBytes <= 0) return null;
                if (numBytes > LengthOfBufferData) return null;

                // build the return buffer
                byte[] retBuffer = new byte[numBytes];
                // copy it in as fast as possible
                Buffer.BlockCopy(netBuffer, 0, retBuffer, 0, numBytes);
                return retBuffer;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the requested amount of data bytes from the buffer, but only peeks
        /// into it and does not remove the returned data
        /// </summary>
        /// <param name="bytePos">the byte position to look at</param>
        /// <returns>the byte at the specified position or 0x00 for fail
        /// Note that 0x00 might also be a legitimate value. So make sure
        /// the required amount of data is in the buffer before calling this</returns>
        /// <history>
        ///    02 Dec 08  Cynic - Originally written
        /// </history>
        public byte PeekByte(int bytePos)
        {
            lock (bufferLock)
            {
                // some sanity checks
                if (netBuffer == null) return 0x00;
                if (netStream == null) return 0x00;
                if (LengthOfBufferData <= 0) return 0x00;
                if (bytePos >= LengthOfBufferData) return 0x00;
                return netBuffer[bytePos];
            }
        }
    }
}
