using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Runtime.InteropServices;
using System.Threading;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Net;
using System.Net.Sockets;

/// +------------------------------------------------------------------------------------------------------------------------------+
///                                                    TERMS OF USE: MIT License                                                  
/// +------------------------------------------------------------------------------------------------------------------------------
/// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation    
/// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,    
/// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
/// is furnished to do so, subject to the following conditions:                                                                   
///                                                                                                                               
/// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
///                                                                                                                               
/// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE          
/// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR         
/// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,   
/// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                         
/// +------------------------------------------------------------------------------------------------------------------------------+

namespace SRV1CSharpConsole
{
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
    /// <summary>
    /// A class to provide a console interface to the SRV1 robot. It connects
    /// to the robot via TCP/IP on the default ports. Once connected, a variety
    /// of pre-defined commands can be sent via button presses or manual commands
    /// can be sent by typing them into a text box. The commands used are those  
    /// from the  
    /// 
    /// The TcpClient and streams programming is directly derived from the 
    /// Microsoft sample code at:
    /// http://msdn.microsoft.com/en-us/library/system.net.sockets.tcpclient.aspx
    /// 
    /// The images on the buttons are taken from the java SRV1Console supplied
    /// by surveyor.com
    /// 
    /// </summary>
    /// <history>
    ///    19 Nov 08  Cynic - Started
    /// </history>
    public partial class frmMain : Form
    {
        // globals vars which hold our TCP/IP connection
        private TcpClient tcpClient = null;
        private TCPBufferedStream tcpBufferedStream = null;

        public delegate SRV1Response MonitorSRV1ResponseDelegate();
        public MonitorSRV1ResponseDelegate MonitorSRV1Response=null;

        public delegate int SendSRV1CommandDelegate(string cmdStr);
        public SendSRV1CommandDelegate SendSRV1Command = null;

        // we record this to try and trap the errors on th 'Q' command
        private string lastNonImageCommandSent=null;

        // this is largest size we can receive in the standard
        // rec buffer. It is sized for a 1280x1024 image plus a 
        // bit to spare"
        private const int MAX_RECEIVE_BUFFER = ((1280 * 1024) + 1024);
        private byte[] stdReceiveBuffer = new Byte[MAX_RECEIVE_BUFFER];

        // this is the total length of the ##IMJxs0s1s2s3 header before
        // the actual image jpg data begins
        private const int IMAGEHEADER_LEN = 10;
        private static string ZDUMPHEADER = "##zdump: ";
        private static string LEAVING_EDITOR = "leaving editor ";

        // the number of msec delay between heartbeats
        private const int HEARTBEAT_DELAYTIME = 100;

        // our heart beat - it manages monitoring actions in a new thread
        private Thread heartBeatThread = null;
        private bool stopHeartBeat = false;

        // the maximum lines in the logfile
        private const int DEFAULT_MAX_LOGFILE_LINES = 200;
        
        private const string DEFAULT_TCPIPADDR = "169.254.0.10";
        private const string DEFAULT_TCPIPPORT = "10001";
        private bool motorInitSent = false;

        private bool wantCameraOutput = false;

        // ccode file read/write options
        private string lastReadOrSavedCCodeFile = null;
        private string currentCCodeFile = null;
        private const string DEFAULT_CCODE_DIR = "CCode Files";
        private const string DEFAULT_CCODE_FILENAME = "test";
        private const string DEFAULT_CCODE_EXT = "c";

        // line editor commands and controls
        private const string ESC_ASHEXSTR = "1B";
        private const string ENTER_EDITMODE_ASSTR = "E";
        private const string ENTER_EDITINSERTLINE_ASSTR = "I";
        private const string EXIT_EDITMODE_ASSTR = "X";
        private const string CLEAR_FLASHBUFFER_ASSTR = "zc";
        private const string GET_FLASHBUFFERCRC_ASSTR = "zC";
        private const string GET_FLASHBUFFERCODE_ASSTR = "zd";
        private const string NOFLASHBUFFER_CONTENTS = @"<no flash buffer contents to display>";
        private const string READFLASHSECTORINTOFLASHBUFFER_ASSTR = "zr";
        private const string WRITEFLASHBUFFERINTOFLASHSECTOR_ASSTR = "zw";

        // delay this long between sending lines or the 
        // line editor gets confused sometimes
        private const int LINE_EDITOR_DELAY_TIME = 100;

        // CCode flash buffer commands
        const string RUN_FLASHBUFFERCODE_ASSTR = "Q";

        const string RESET_BLACKFIN_ASSTR = "$!";

        // motor control defines
        private const string MOTOR_BACK_ASSTR = "2";
        private const string MOTOR_STOP_ASSTR = "5";
        private const string MOTOR_FORWARD_ASSTR = "8";
        private const string LASER_ON_ASSTR = "l";
        private const string LASER_OFF_ASSTR = "L";

        private const string IMAGE_SIZE_160x120_ASSTR = "a";
        private const string IMAGE_SIZE_320x240_ASSTR = "b";
        private const string IMAGE_SIZE_640x480_ASSTR = "c";
        private const string IMAGE_SIZE_1280x1024_ASSTR = "d";
        private const string IMAGE_FETCH_ASSTR = "I";

        private const string MOTOR_DRIFTLEFT_ASHEXSTR = "4D 28 3C 00";
        private const string MOTOR_DRIFTRIGHT_ASHEXSTR = "4D 3C 28 00";
        private const string MOTOR_TURNLEFT_ASHEXSTR = "4D 1E 46 00";
        private const string MOTOR_TURNRIGHT_ASHEXSTR = "4D 46 1E 00";
        private const string MOTOR_BACKLEFT_ASHEXSTR = "4D E2 BA 00";
        private const string MOTOR_BACKRIGHT_ASHEXSTR = "4D BA E2 00";
        private const string MOTOR_TURNCW_ASHEXSTR = "4D 40 C0 14";
        private const string MOTOR_TURNCCW_ASHEXSTR = "4D C0 40 14";

        // this is a standard Mxxx command sets both motors to run slowly forward
        // for the smallest possible time. 
        private const string DEFAULT_MOTOR_INIT_ASHEXSTR = "4D 00 00 01";

        // a sample C program
        const string DEMO_PROGRAM = "/* A sample C Code file, it runs the motors forward for 1 sec */\nmain()\n{\n    motors(50, 50);\n    delay(1000);\n    motors(0, 0);\n}";

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// The constructor
        /// </summary>
        /// <history>
        ///    19 Nov 08  Cynic - Originally written
        /// </history>
        public frmMain()
        {
            InitializeComponent();

            // set this now - we call this from other threads to monitor
            // the responses from the SRV1
            MonitorSRV1Response = new MonitorSRV1ResponseDelegate(HandleSRV1Response);
            SendSRV1Command = new SendSRV1CommandDelegate(SendSRV1CommandAsStr);

            textBoxIPAddress.Text=DEFAULT_TCPIPADDR;
            textBoxIPPort.Text=DEFAULT_TCPIPPORT;

            // set this now so the user can try something out
            richTextBoxCCode.Text = DEMO_PROGRAM;

            // set our heartbeat going
            LaunchHeartBeat();

            // set this up to reflect reality
            SyncConnectionStatusDisplay();
        }

        // ########################################################################
        // ##### SendReceiveSRV1Code - code which sends/receives commands to/from SRV1
        // ########################################################################

        #region SendReceiveSRV1Code

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Sends a command to run the contents of the SRV1 flash buffer. Performs 
        /// appropriate checks first
        /// </summary>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    28 Nov 08  Cynic - Originally written
        /// </history>
        private int RunSRV1Buffer()
        {
            bool originalWantCameraOutput = false;
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return 100;
            }

            // clear this out
            ClearErrorWindow();

            // normally we would just send a 'Q' command and
            // be done with it. However, if the flash buffer
            // is empty the 'Q' command just locks up the SRV1
            // so we check to see if there is anything in the
            // flash buffer before we send the command (sigh).
            if (DoesFlashBufferHaveContents() == false)
            {
                // no contents
                DiagnosticDialog("The flash buffer seems to be empty.");
                return 101;
            }
            try
            {
                // stop the camera
                originalWantCameraOutput = wantCameraOutput;
                wantCameraOutput = false;
                Thread.Sleep(HEARTBEAT_DELAYTIME);
                // send a Q command to run whatever is in there
                SendSRV1CommandAsStr(RUN_FLASHBUFFERCODE_ASSTR);
            }
            finally
            {
                wantCameraOutput = originalWantCameraOutput;
            }
        return 0;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Checks to see if the flash buffer has any contents. We do this by
        /// seeing if the CRC is zero or not
        /// </summary>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    28 Nov 08  Cynic - Originally written
        /// </history>
        private bool DoesFlashBufferHaveContents()
        {
            bool originalWantCameraOutput = false;
            const string ZEROCRCRESPONSE = "##zCRC: 0x0000";

            if (this.IsConnected == false) return false;

            // make sure we have nothing sitting in the 
            // receive buffer
            HandleSRV1Response();
            try
            {
                // stop the camera
                originalWantCameraOutput = wantCameraOutput;
                wantCameraOutput = false;
                Thread.Sleep(HEARTBEAT_DELAYTIME);
                // get the CRC of the flash buffer contents
                SendSRV1CommandAsStr(GET_FLASHBUFFERCRC_ASSTR);
                Thread.Sleep(LINE_EDITOR_DELAY_TIME);
                // handle the response ourselves rather than letting
                // the heartbeat do it. This cleans it out
                SRV1Response responseObj = HandleSRV1Response();
                if (responseObj == null) return false;
                if (responseObj.ResponseType != SRVResponseTypeEnum.STD_1CHAR_AND_BINARY) return false;
                if (responseObj.ResponseDataAsString == null) return false;
                if (responseObj.ResponseDataAsString == ZEROCRCRESPONSE) return false;
            }
            finally
            {
                wantCameraOutput = originalWantCameraOutput;
            }

            return true;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Fetches the contents of the SRV1 flash buffer
        /// </summary>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    28 Nov 08  Cynic - Originally written
        /// </history>
        private string GetSRV1FlashBufferContents()
        {
            bool originalWantCameraOutput = false;
            SRV1Response responseObj = null;
            // make sure we have nothing sitting in the 
            // receive buffer
            HandleSRV1Response();
            try
            {
                // stop the camera
                originalWantCameraOutput = wantCameraOutput;
                wantCameraOutput = false;
                Thread.Sleep(HEARTBEAT_DELAYTIME);
                // get the flash buffer contents
                SendSRV1CommandAsStr(GET_FLASHBUFFERCODE_ASSTR);
                Thread.Sleep(LINE_EDITOR_DELAY_TIME);
                // handle the response ourselves rather than letting
                // the heartbeat do it. This cleans it out
                responseObj = HandleSRV1Response();
                if (responseObj == null) return NOFLASHBUFFER_CONTENTS;
                if (responseObj.ResponseType != SRVResponseTypeEnum.ZDUMP) return NOFLASHBUFFER_CONTENTS;
                if (responseObj.ResponseDataAsString == null) return NOFLASHBUFFER_CONTENTS;
                if (responseObj.ResponseDataAsString.Length == 0) return NOFLASHBUFFER_CONTENTS;
            }
            finally
            {
                wantCameraOutput = originalWantCameraOutput;
            }

            return responseObj.ResponseDataAsString;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Sends a multiline string to the SRV1 and stores it in the flash
        /// buffer. This proc uses the line editor to do this - essentially
        /// just mimicing how the lines could be typed in manually.
        /// </summary>
        /// <param name="byteArr">The byte array to send</param>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    27 Nov 08  Cynic - Originally written
        /// </history>
        private void SendCodeByLineEditor(string codeStr)
        {
            string[] splitArr = null;
            char[] splitter = { '\n' };
            bool originalWantCameraOutput = false;

            // also need to check this
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }

            // sanity check
            if ((codeStr == null) || (codeStr.Length == 0))
            {
                DiagnosticDialog("There is no C code to send to the SRV1 in the panel.");
                return;
            }

            try
            {
                // stop the camera
                originalWantCameraOutput = wantCameraOutput;
                wantCameraOutput = false;
                Thread.Sleep(HEARTBEAT_DELAYTIME);

                // remove any \r chars and split it on the \n chars
                string workingStr = codeStr.Replace("\r", "");
                splitArr = workingStr.Split(splitter);

                // clean out the flash buffer
                SendSRV1CommandAsStr(CLEAR_FLASHBUFFER_ASSTR);
                Thread.Sleep(LINE_EDITOR_DELAY_TIME);
                // handle the response ourselves rather than letting
                // the heartbeat do it. This cleans it out
                HandleSRV1Response();

                // put the SRV1 in edit mode
                SendSRV1CommandAsStr(ENTER_EDITMODE_ASSTR);
                Thread.Sleep(LINE_EDITOR_DELAY_TIME);

                // send each string in turn
                foreach (string outStr in splitArr)
                {
                    // send I insert command to the line editor
                    SendSRV1CommandAsStr(ENTER_EDITINSERTLINE_ASSTR);
                    // send the line
                    SendSRV1CommandAsStr(outStr);
                    // take the line editor out of insert mode
                    SendSRV1CommandAsHexStr(ESC_ASHEXSTR);
                    Thread.Sleep(LINE_EDITOR_DELAY_TIME);
                }
                Thread.Sleep(LINE_EDITOR_DELAY_TIME);
                // take it out of edit mode
                SendSRV1CommandAsStr(EXIT_EDITMODE_ASSTR);
            }
            finally
            {
                wantCameraOutput = originalWantCameraOutput;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Sends a command to control the motors - performs appropriate checks
        /// Will send an Mxxx init command if not already sent
        /// </summary>
        /// <param name="motorCommand">The motor command to send</param>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private int SendSimpleMotorCommand(string motorCommand)
        {
            // sanity check
            if ((motorCommand == null) || (motorCommand.Length == 0))
            {
                // should never happen
                DiagnosticDialog("SendMotorCommand: Bad Input Param");
                return 100;
            }

            // we must be on the form thread!!!
            if (InvokeRequired == true)
            {
                // only call from the form thread 
                return 1011;
            }

            // also need to check this
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return 101;
            }
            // the SRV1 requires an Mxxx command to activate its motors 
            // before the simpler control commands become active
            if (motorInitSent == false)
            {
                SendSRV1CommandAsHexStr(DEFAULT_MOTOR_INIT_ASHEXSTR);
                motorInitSent = true;
            }
            return SendSRV1CommandAsStr(motorCommand);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Sends a hex string command to the SRV1. Note that the byte
        /// to send should be specified as twochar ascii hex values with
        /// space separation. For example "4D E2 BA 00"
        /// </summary>
        /// <param name="cmdStr">The command to send</param>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private int SendSRV1CommandAsHexStr(string cmdStr)
        {
            if ((cmdStr == null) || (cmdStr.Length == 0))
            {
                // should never happen
                return 221;
            }

            // we must be on the form thread!!!
            if (InvokeRequired == true)
            {
                // only call from the form thread 
                return 2011;
            }

            // also need to check this
            if (this.IsConnected == false)
            {
                return 222;
            }
            LogMessage("SRV1 send:>" + cmdStr);

            byte[] cmdArr = ConvertSpacedHexStringToByteArray(cmdStr);
            return SendSRV1CommandAsByteArr(cmdArr);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Sends a string command to the SRV1. Note that the byte
        /// to send should be specified as single ascii values with
        /// no separation otherwise the separator gets sent as well.
        /// </summary>
        /// <param name="cmdStr">The command to send</param>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private int SendSRV1CommandAsStr(string cmdStr)
        {
            if ((cmdStr == null) || (cmdStr.Length == 0))
            {
                // should never happen
                return 331;
            }

            // we must be on the form thread!!!
            if (InvokeRequired == true)
            {
                // only call from the form thread using SendSRV1Command
                // See the HeartBeat() for syntax
                return 3011;
            }

            // also need to check this
            if (this.IsConnected == false)
            {
                return 333;
            }
            // should we log this, only if it is a 
            if (checkBoxNoShowIMGMessages.Checked == false)
            {
                // log everything
                LogMessage("SRV1 send:>" + cmdStr);
            }
            else
            {
                if (cmdStr != IMAGE_FETCH_ASSTR)
                {
                    // not an image fetch command, log it
                    LogMessage("SRV1 send:>" + cmdStr);
                    lastNonImageCommandSent = cmdStr;
                }
            }
            byte[] byData = System.Text.Encoding.ASCII.GetBytes(cmdStr);
            return SendSRV1CommandAsByteArr(byData);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Sends an array of bytes to the SRV1
        /// </summary>
        /// <param name="byteArr">The byte array to send</param>
        /// <returns>z - success, nz fail</returns>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private int SendSRV1CommandAsByteArr(byte[] byteArr)
        {
            if ((byteArr == null) || (byteArr.Length == 0))
            {
                // should never happen
                return 441;
            }

            // we must be on the form thread!!!
            if (InvokeRequired == true)
            {
                // only call from the form thread
                return 4011;
            }

            // also need to check this
            if (this.IsConnected == false)
            {
                return 444;
            }
           // tcpStream.Write(byteArr, 0, byteArr.Length);
            tcpBufferedStream.NetStream.Write(byteArr, 0, byteArr.Length);
            return 0;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A handler for SRV1 responses. We can call this out of the heartbeat
        /// or individually if we want to wait for SRV1 responses. Because 
        /// we force everything onto the form thread with the invoke we do not
        /// need to acquire locks. Only the form thread can do usefule work here
        /// </summary>
        /// <param name="mSecWaitTime">The time out in msec, this is fed to 
        /// the socket as the timeout value</param>
        /// <returns>A SRV1Response class. Note that reponses are always logged
        /// and if it is a returning image response it is displayed in the 
        /// image panel. If multiple responses are present only the last 
        /// response will be returned. The others will be consumed.
        /// 
        /// -555 - not connected
        /// -556 - no data
        /// -99  - response data invalid or uninterpretable
        /// 
        /// </returns>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        /// </history>
        private SRV1Response HandleSRV1Response()
        {
            // set this up
            Int32 bytes = 0;

            // You do not want to be calling this from any other thread
            // than the form message pump do you? no, no, no, I assure
            // you that you do not. 
            if (InvokeRequired==true)
            {
                // we are now sure we are on the proper thread!
                return (SRV1Response)this.Invoke(MonitorSRV1Response);
            }

            // also need to check this
            if (this.IsConnected == false)
            {
                // this is a standard not connected error
                return new SRV1Response(-555);
            }

            // Read the first batch of the TcpServer response bytes.
            bytes = tcpBufferedStream.ConsumeStreamData();
            // any data there?
            if(bytes==0)
            {
                // no there is not
                return new SRV1Response(-556);
            }

//LogMessage("reponse bytes="+bytes.ToString());

            // this does everything necessary
            return ProcessSRV1Response(tcpBufferedStream);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Processes the reponse from the SRV1. Usually this just means logging the 
        /// incoming commands but in the case of image data it means displaying
        /// the returning image
        /// </summary>
        /// <remarks>Is recursive. If there is more than one response in the 
        /// buffer this proc will call itself to process it. It delays processing
        /// until after the recursion so the events happen in the order the 
        /// SRV1 sent them. See the comments below.</remarks>
        /// <returns>A populated SRV1Response object
        /// errorCode=-99 for unknown error, 
        /// zero bytes on the dataLen assumes a timeout and no data
        /// </returns>
        /// <param name="bufferedStream">our buffered stream object to consume process
        /// the response from</param>
        /// <history>
        ///    24 Nov 08  Cynic - Originally written
        ///    02 Dec 08  Cynic - converted to process buffered tcp streams so as not
        ///                       to loose data with big slow packets
        /// </history>
        private SRV1Response ProcessSRV1Response(TCPBufferedStream bufferedStream)
        {
            MemoryStream sTmp = null;
            Bitmap bTmp = null;

            // set up our response object to a unknown error
            SRV1Response responseObj=new SRV1Response(-99,SRVResponseTypeEnum.ERROR);

            if (InvokeRequired == true)
            {
                // we are now sure we are on the proper thread!
                return responseObj;
            }

            // some sanity checks
            if (bufferedStream == null) return responseObj;
            if (bufferedStream.DataAvailable == false)
            {
                // nothing returned - assume timed out, set up response
                responseObj.ErrorCode = 0;
                responseObj.ResponseType = SRVResponseTypeEnum.TIMED_OUT;
                return responseObj;
            }
            else if (bufferedStream.LengthOfBufferData < 2) 
            {
                // we do not know what the problem is here
                responseObj.ErrorCode = 0;
                responseObj.ResponseType = SRVResponseTypeEnum.UNKNOWN;
                int bufLen=bufferedStream.LengthOfBufferData;
                responseObj.SetResponseData(bufferedStream.GetDataBytes(bufLen), 0, bufLen);
                // clean up - zero the buffer
                bufferedStream.ConsumeStreamData();
                bufferedStream.ResetBuffer();
                // note, we fall through here, there is no return
            }
            // is the first character a '#' char?
            else if (bufferedStream.PeekByte(0) != (byte)'#')
            {
                // no it is not, have we got exit editor signal? this is leaving "editor \n"
                if ((bufferedStream.LengthOfBufferData >= LEAVING_EDITOR.Length) &&
                    (bufferedStream.PeekByte(2) == (byte)LEAVING_EDITOR[2]) &&
                    (bufferedStream.PeekByte(4) == (byte)LEAVING_EDITOR[4]) &&
                    (bufferedStream.PeekByte(6) == (byte)LEAVING_EDITOR[6]) &&
                    (bufferedStream.PeekByte(8) == (byte)LEAVING_EDITOR[8]) &&
                    (bufferedStream.PeekByte(10) == (byte)LEAVING_EDITOR[10]) &&
                    (bufferedStream.PeekByte(11) == (byte)LEAVING_EDITOR[11]) &&
                    (bufferedStream.PeekByte(12) == (byte)LEAVING_EDITOR[12]))
                {
                    // we do not know what this is, we cannot cope with this
                    responseObj.ErrorCode = 0;
                    responseObj.ResponseType = SRVResponseTypeEnum.STD_TEXTONLY;
                    int countToNewLineChar = bufferedStream.LocateByte((byte)'\n');
                    if (countToNewLineChar <= 0)
                    {
                        responseObj.SetResponseData(bufferedStream.GetDataBytes(LEAVING_EDITOR.Length), 0, LEAVING_EDITOR.Length);
                    }
                    else
                    {
                        responseObj.SetResponseData(bufferedStream.GetDataBytes(countToNewLineChar), 0, countToNewLineChar);
                        // consume the '\n' text - we do not need it
                        bufferedStream.GetDataBytes(1);
                    }
                    // note, we fall through here, there is no return
                }
                else
                {
                    // we do not know what this is, we cannot cope with this
                    responseObj.ErrorCode = 0;
                    responseObj.ResponseType = SRVResponseTypeEnum.UNKNOWN;
                    int bufLen = bufferedStream.LengthOfBufferData;
                    responseObj.SetResponseData(bufferedStream.GetDataBytes(bufLen), 0, bufLen);
                    // clean up - zero the buffer
                    bufferedStream.ConsumeStreamData();
                    bufferedStream.ResetBuffer();
                    // note, we fall through here, there is no return
                }
            }
            // is the second character a '#' char?
            else if (bufferedStream.PeekByte(1) != (byte)'#')
            {
                // no it is not - this is probably just a simple #? response
                responseObj.ErrorCode=0;
                responseObj.ResponseType=SRVResponseTypeEnum.STD_1CHAR_TEXT;
                responseObj.SetResponseData(bufferedStream.GetDataBytes(2), 0, 2);
                // note, we fall through here, there is no return
            }
            else
            {
                // if we get here we have a "##" pair leading the response. This
                // is the SRV1's way of saying that extended maybe binary data is 
                // follows the next digit. The format is response specific

                // have we got returning image data? this is ##IMG...
                if ((bufferedStream.LengthOfBufferData > IMAGEHEADER_LEN) && (bufferedStream.PeekByte(2) == (byte)'I') && (bufferedStream.PeekByte(3) == (byte)'M') && (bufferedStream.PeekByte(4) == (byte)'J'))
                {
                    try
                    {
                        // convert the size encoded in the returned byte []
                        int jpgSize = ConvertFourBytesToScaledInt(bufferedStream.PeekByte(6), bufferedStream.PeekByte(7), bufferedStream.PeekByte(8), bufferedStream.PeekByte(9));
                        if (bufferedStream.LengthOfBufferData < jpgSize + IMAGEHEADER_LEN)
                        {
                            // this probably means we got an incomplete transmission because we started
                            // reading before the full image data got delivered. This happens a lot
                            // for large images and a fairly quick heartbeat. We just continue on here
                            // and pick it up on the next heartbeat
                            responseObj.ErrorCode = 0;
                            responseObj.ResponseType = SRVResponseTypeEnum.TIMED_OUT;
                            return responseObj;
                        }
                        // else populate the real response object
                        responseObj.ErrorCode = 0;
                        responseObj.ResponseType = SRVResponseTypeEnum.STD_IMAGE;
                        // consume the image header - we do not need it
                        bufferedStream.GetDataBytes(IMAGEHEADER_LEN);
                        // now store the actual image data
                        responseObj.SetResponseData(bufferedStream.GetDataBytes(jpgSize), 0, jpgSize);

                        // hand the image over to the picture box. Note we don't
                        // have to do any processing here beside just shoveling the
                        // data over the wall with the help of the ever handy MemoryStream
                        try
                        {
                            sTmp = new MemoryStream(responseObj.ResponseDataAsByteArray);
                            bTmp = new Bitmap(sTmp);
                            pictureBoxCameraOutput.Image = bTmp;
                        }
                        finally
                        {
                            if (sTmp != null)
                            {
                                sTmp.Close();
                                sTmp = null;
                            }
                            bTmp = null;
                        }
                    }
                    catch (Exception)
                    {
                    }
                    // note, we fall through here, there is no return
                }
                // have we got returning zdump data? this is ##zdump: \n {...} where {...} is the
                // buffer contents without newlines.
                else if ((bufferedStream.LengthOfBufferData > ZDUMPHEADER.Length) &&
                    (bufferedStream.PeekByte(2) == (byte)ZDUMPHEADER[2]) &&
                    (bufferedStream.PeekByte(3) == (byte)ZDUMPHEADER[3]) &&
                    (bufferedStream.PeekByte(4) == (byte)ZDUMPHEADER[4]) &&
                    (bufferedStream.PeekByte(5) == (byte)ZDUMPHEADER[5]) &&
                    (bufferedStream.PeekByte(6) == (byte)ZDUMPHEADER[6]) &&
                    (bufferedStream.PeekByte(7) == (byte)ZDUMPHEADER[7]) &&
                    (bufferedStream.PeekByte(8) == (byte)ZDUMPHEADER[8]))
                {
                    int bufLen = bufferedStream.LengthOfBufferData;
                    int countToNewLineChar = bufferedStream.LocateByte((byte)'\n');
                    if (countToNewLineChar <= 0)
                    {
                        // should never happen
                        // return an error response. The '\n' should be there
                        responseObj.ErrorCode = 0;
                        responseObj.ResponseType = SRVResponseTypeEnum.ERROR;
                        // not found, clean up - zero the buffer
                        bufferedStream.ConsumeStreamData();
                        bufferedStream.ResetBuffer();
                    }
                    if (countToNewLineChar >= bufLen)
                    {
                        // empty buffer?
                        // return an error response. The '\n' should be there
                        responseObj.ErrorCode = 0;
                        responseObj.ResponseType = SRVResponseTypeEnum.ZDUMP;
                        // clean up - zero the buffer
                        bufferedStream.ConsumeStreamData();
                        bufferedStream.ResetBuffer();
                    }
                    else
                    {
                        // the '\n char was found, this will be the '\n' trailing the "##zdump: \n" ZDUMPHEADER
                        // we consume the header and put the remainder in the response object. Now at this point
                        // we cannot be sure that only the dump data is present. A second \n or any termination
                        // character is not provided. We just tip the lot into the response object and hope that
                        // no packets from other sources (ie image responses) have managed to append themselves
                        // before this packet got read.
                        responseObj.ErrorCode = 0;
                        responseObj.ResponseType = SRVResponseTypeEnum.ZDUMP;
                        // consume the "##zdump:\n" ZDUMPHEADER text - we do not need it
                        bufferedStream.GetDataBytes(countToNewLineChar);
                        // consume the '\n' text - we do not need it
                        bufferedStream.GetDataBytes(1);
                        int newBufLen = bufferedStream.LengthOfBufferData;
                        // here we are pretty sure we have reasonable data
                        responseObj.SetResponseData(bufferedStream.GetDataBytes(newBufLen), 0, newBufLen);
                    }
                }
                else
                {
                    // In here we know we have a ## leader
                    // assume responseObj of one or more indicator chars and some (probably binary) terminated 
                    // with a '\n' char - the only one that is not is the ##IMG and that is handled above
                    int countToNewLineChar = bufferedStream.LocateByte((byte)'\n');
                    if (countToNewLineChar <= 0)
                    {
                        // return an error response. The '\n' should be there
                        responseObj.ErrorCode = 0;
                        responseObj.ResponseType = SRVResponseTypeEnum.ERROR;
                        // not found, clean up - zero the buffer
                        bufferedStream.ConsumeStreamData();
                        bufferedStream.ResetBuffer();
                    }
                    else
                    {
                        // the '\n char was found, put the returned values in the responseObj
                        responseObj.ErrorCode = 0;
                        responseObj.ResponseType = SRVResponseTypeEnum.STD_1CHAR_AND_BINARY;
                        responseObj.SetResponseData(bufferedStream.GetDataBytes(countToNewLineChar), 0, countToNewLineChar);
                        // consume the '\n' char - we do not need it
                        bufferedStream.GetDataBytes(1);
                    }
                    // note, we fall through here, there is no return
                }
            }

            // here we log the response - if appropriate. Note that because this
            // happens before the recursion returns we will have processed the 
            // responses in the order they were sent.

            // also note that although we process all of the responses
            // the only one to get returned to the caller is the most recent
            // one sent by the SRV1
            switch(responseObj.ResponseType)
            {
                case SRVResponseTypeEnum.STD_1CHAR_TEXT:
                {
                    LogMessage("SRV1(a) says:>" + responseObj.ResponseDataAsString);
                    break;
                }
                case SRVResponseTypeEnum.STD_1CHAR_AND_BINARY:
                {
                    // we do not know how to deal with this - just output first 
                    // 20 chars converting anything not printable to '?' chars
                    StringBuilder sb = new StringBuilder();
                    if (responseObj.ResponseDataAsByteArray == null)
                    {
                        LogMessage("SRV1(b) says:>unknown response type(1)");
                        break;
                    }
                    // convert each byte and add to our string builder
                    foreach (byte tmpByte in responseObj.ResponseDataAsByteArray)
                    {
                        char tmpChar = Convert.ToChar(tmpByte);
                        if ((tmpChar < ' ') || (tmpChar > '~'))
                        {
                            // use this for unprintable chars
                            sb.Append("~");
                        }
                        else
                        {
                            sb.Append(tmpChar);
                        }
                    }
                    LogMessage("SRV1(c) says:>" + sb.ToString());
                    break;
                }
                case SRVResponseTypeEnum.ZDUMP:
                {
                    // this is the output of a zd command
                    StringBuilder sb = new StringBuilder();
                    if (responseObj.ResponseDataAsByteArray == null)
                    {
                        LogMessage("SRV1:>" + ZDUMPHEADER);
                        break;
                    }
                    // convert each byte and add to our string builder
                    foreach (byte tmpByte in responseObj.ResponseDataAsByteArray)
                    {
                        char tmpChar = Convert.ToChar(tmpByte);
                        if ((tmpChar < ' ') || (tmpChar > '~'))
                        {
                            // use this for unprintable chars
                            sb.Append("~");
                        }
                        else
                        {
                            sb.Append(tmpChar);
                        }
                    }
                    LogMessage("SRV1 says:>" + ZDUMPHEADER);
                    LogMessage(sb.ToString());
                    break;
                }
                case SRVResponseTypeEnum.STD_TEXTONLY:
                {
                    // this is some sort of text output
                    StringBuilder sb = new StringBuilder();
                    if (responseObj.ResponseDataAsByteArray == null)
                    {
                        LogMessage("SRV1 says:>Unknown Text");
                        break;
                    }
                    else
                    {
                        LogMessage("SRV1 says:>" + responseObj.ResponseDataAsString);
                        break;
                    }
                 }
                case SRVResponseTypeEnum.UNKNOWN:
                {
                    // we do not know how to deal with this - just output first 
                    // 20 chars converting anything not printable to '?' chars
                    StringBuilder sb = new StringBuilder();
                    if (responseObj.ResponseDataAsByteArray == null)
                    {
                        LogMessage("SRV1(d) says:>unknown response type(2)");
                        break;
                    }
                    else if ((lastNonImageCommandSent != null) && (lastNonImageCommandSent == RUN_FLASHBUFFERCODE_ASSTR))
                    {
                        // apparently the last command sent was a "Q" to run the flash buffer.
                        // The SRV1 C interpreter will return errors as free text in this event
                        // for example: "not a variable in line        1~main(){    xotors(semicolon expected in line        1~main(){    xotors"
                        // these can be seen in the log but in the interests of utility it might be
                        // useful to trap these and display them in a panel on the CCode tab and that
                        // is just what the code below does
                        string errStr = responseObj.ResponseDataAsString;
                        // attempt to not barf returning image data out to the
                        // error panel, this implicitly assumes the returning
                        // error never contains more than this many unprintable 
                        // characters. if it does then it'll still be in the log
                        if (CountNonPrintableCharsInString(errStr) < 100)
                        {
                            richTextBoxCCodeErrors.Text = errStr;
                            MessageBeep(0);
                        }
                        // we only accept one of these per "Q" command so reset this
                        // so the next unknown cannot trigger it
                        lastNonImageCommandSent = null;
                        // fall through below so we also pump it out to the log
                    }
                    // special case when we exit the line editor it will 
                    // pump out some help text and the code we just entered
                    // this text starts with "(T)op" we look for this and 
                    // pump it out as is since replacing the \n with ~
                    // chars makes it look like a dogs breakfast
                    byte[] tmpBytes = responseObj.ResponseDataAsByteArray;
                    if ((tmpBytes[0] == (byte)'(') &&
                        (tmpBytes[1] == (byte)'T') &&
                        (tmpBytes[2] == (byte)')') &&
                        (tmpBytes[3] == (byte)'o') &&
                        (tmpBytes[4] == (byte)'p'))
                    {
                        foreach (byte tmpByte in tmpBytes)
                        {
                            char tmpChar = Convert.ToChar(tmpByte);
                            sb.Append(tmpChar);
                        }
                    }
                    else
                    {
                        // convert each byte and add to our string builder
                        foreach (byte tmpByte in tmpBytes)
                        {
                            char tmpChar = Convert.ToChar(tmpByte);
                            if ((tmpChar < ' ') || (tmpChar > '~'))
                            {
                                // use this for unprintable chars
                                sb.Append("~");
                            }
                            else
                            {
                                sb.Append(tmpChar);
                            }
                        }
                    }
                    LogMessage("SRV1(e) says:>" + sb.ToString());
                    break;
                }
                case SRVResponseTypeEnum.STD_IMAGE:
                {
                    // are we allowed to report these
                    if (checkBoxNoShowIMGMessages.Checked == false)
                    {
                        // we just say we got one, not going to spatter
                        // all sorts of binary to log
                        LogMessage("SRV1(f) says:>##IMG...");
                    }
                    break;
                }
                default:
                {
                    LogMessage("SRV1(g)>Unknown response type");
                    break;
                }
            }

            // recursively call ourselves - if necessary. If our dataOffset
            // is still less than the length we can consume more. The restriction
            // on bufferedStream.LengthOfBufferData>0 is just defensive programming 
            // to avoid endless loops
            if (bufferedStream.LengthOfBufferData>0)
            {
                // recursively call ourselves. Note we ignore the return value here
                // we only ever return the latest one to the caller
                ProcessSRV1Response(bufferedStream);

            }
            return responseObj;
        }
        #endregion


        // ########################################################################
        // ##### Code to read and write exclusion list files
        // ########################################################################
        #region ReadAndWriteFiles

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Fetches and processes the contents of the richTextBoxCCode.Text 
        /// field in a way consistent for storage
        /// </summary>
        /// <returns>the processed code text or "" for fail</returns>
        /// <history>
        ///   04 Dec 08  Cynic - Originally written
        /// </history>
        private string GetCCodeTextInConsistentFormat()
        {
            if (richTextBoxCCode.Text == null) return "";
            // we make all of the line terminators "\r\n" 
            return richTextBoxCCode.Text.Replace("\r", "").Replace("\n", "\r\n");
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Reads a file into the CCode panel
        /// </summary>
        /// <history>
        ///   05 Dec 08  Cynic - Originally written
        /// </history>
        private void ReadFile()
        {
            string defaultDirectory = null;
            string defaultFileName = null;

            // set this
            string filePathAndNameToOpen = null;

            // try to figure out the most recent file to get
            if (currentCCodeFile != null)
            {
                // pick the directory off of the current file
                defaultDirectory = Path.GetDirectoryName(lastReadOrSavedCCodeFile);
            }
            else if (lastReadOrSavedCCodeFile != null)
            {
                // pick the directory off of the last used file
                defaultDirectory = Path.GetDirectoryName(lastReadOrSavedCCodeFile);
            }
            else
            {
                // build a path out of the app startup directory and the default dir
                defaultDirectory = Path.Combine(Application.StartupPath, DEFAULT_CCODE_DIR);
            }

            // now sort out a filename
            if (currentCCodeFile != null)
            {
                defaultFileName = Path.GetFileNameWithoutExtension(currentCCodeFile);
            }
            else if (lastReadOrSavedCCodeFile != null)
            {
                defaultFileName = Path.GetFileNameWithoutExtension(lastReadOrSavedCCodeFile);
            }
            // the defaultFileName can be null here

            OpenFileDialog openFileDialog1 = new OpenFileDialog();

            // NOTE: if we do not have the initial directory here just ignore
            // the operating system seems to remember it (on XP at least)
            if (defaultDirectory != null) openFileDialog1.InitialDirectory = defaultDirectory;
            if (defaultFileName != null) openFileDialog1.FileName = defaultFileName;
            openFileDialog1.Filter = "CCode files (*." + DEFAULT_CCODE_EXT + ")|*." + DEFAULT_CCODE_EXT;
            openFileDialog1.FilterIndex = 1;
            openFileDialog1.RestoreDirectory = true;

            if (openFileDialog1.ShowDialog() != DialogResult.OK)
            {
                // user must have cancelled. Thats OK we just leave now
                LogMessage("OpenOfFile Cancelled");
                return;
            }

            // reset these
            richTextBoxCCode.Text = "";
            lastReadOrSavedCCodeFile = null;
            currentCCodeFile = null;

            filePathAndNameToOpen = openFileDialog1.FileName;
            // did we get one
            if ((filePathAndNameToOpen == null) || (filePathAndNameToOpen.Length==0))
            {
                DiagnosticDialog("No file name specified.");
            }

            // read the file
            try
            {
                richTextBoxCCode.Text = File.ReadAllText(filePathAndNameToOpen);
                // if we get here there was no exception
                lastReadOrSavedCCodeFile = filePathAndNameToOpen;
                currentCCodeFile = filePathAndNameToOpen;
                DiagnosticDialog("The file has now been opened.");
            }
            catch (Exception ex)
            {
                // let the user know
                DiagnosticDialog(ex);
            }

            // record this
            return;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Saves the current ccode panel contents to the existing file name. If
        /// the filename does not exist the user will be prompted
        /// </summary>
        /// <history>
        ///   03 Dec 08  Cynic - Originally written
        /// </history>
        private void SaveFile()
        {
            // do we have a current filename? if not save it as
            if ((currentCCodeFile == null) || (currentCCodeFile.Length == 0))
            {
                SaveFileAs();
                return;
            }

            // prepare the ccCode text so that it can be saved
            // this is essentially just linefeed conversions.
            string cCode = GetCCodeTextInConsistentFormat();
            if ((cCode == null) || (cCode.Length == 0))
            {
                DiagnosticDialog("The CCode window appears to be empty.");
                return;
            }

            // write the file
            try
            {
                File.WriteAllText(currentCCodeFile, richTextBoxCCode.Text);
                // if we get here there was no exception
                lastReadOrSavedCCodeFile = currentCCodeFile;
                DiagnosticDialog("The file has now been saved.");
            }
            catch (Exception ex)
            {
                // let the user know
                DiagnosticDialog(ex);
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Saves the current ccode panel contents and always pPrompts a user for 
        /// the name under which to save the CCode file. Will warn if the file
        /// already exists
        /// </summary>
        /// <history>
        ///   03 Dec 08  Cynic - Originally written
        /// </history>
        private void SaveFileAs()
        {
            int retInt;
            string ccodeFileName = null;

            retInt = GetSaveNameForCCodeFile(ref ccodeFileName);
            if (retInt != 0)
            {
                DiagnosticDialog("Save of ccode file cancelled by user request.");
                return;
            }
            if ((ccodeFileName == null) || (ccodeFileName.Length == 0))
            {
                DiagnosticDialog("Bad or no file name returned.\n\nFile cannot be saved.");
                return;
            }
            // prepare the ccCode text so that it can be saved
            // this is essentially just linefeed conversions.
            string cCode = GetCCodeTextInConsistentFormat();
            if ((cCode == null) || (cCode.Length == 0))
            {
                DiagnosticDialog("The CCode window appears to be empty.");
                return;
            }

            // does the file already exist - we really should warn the user
            if (File.Exists(ccodeFileName))
            {
                DialogResult dlgRes = MessageBox.Show("The file\n\n" + ccodeFileName + "\n\nalready exists. Do you want to overwrite it?", "File already exists", MessageBoxButtons.YesNo, MessageBoxIcon.Question);
                if (dlgRes != DialogResult.Yes)
                {
                    DiagnosticDialog("The save of the file has been cancelled.");
                    return;
                }
            }

            // write the file
            try
            {
                File.WriteAllText(ccodeFileName, cCode);
                // if we get here there was no exception
                lastReadOrSavedCCodeFile = ccodeFileName;
                currentCCodeFile = ccodeFileName;
                DiagnosticDialog("The file has now been saved.");
            }
            catch (Exception ex)
            {
                // let the user know
                DiagnosticDialog(ex);
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Prompts a user for the name under which to save the CCode file
        /// </summary>
        /// <param name="ccodeListFileName">the file name and path or null</param>
        /// <returns>z success, nz failed</returns>
        /// <history>
        ///   03 Dec 08  Cynic - Originally written
        /// </history>
        private int GetSaveNameForCCodeFile(ref string ccodeFileName)
        {
            string defaultDirectory=null;
            string defaultFileName=null;

            ccodeFileName = null;

            // by the time we get here we either have a suggested name or we do not
            if (currentCCodeFile != null)
            {
                // pick the directory off of the current file
                defaultDirectory = Path.GetDirectoryName(lastReadOrSavedCCodeFile);
            }
            else if (lastReadOrSavedCCodeFile != null)
            {
                // pick the directory off of the last used file
                defaultDirectory = Path.GetDirectoryName(lastReadOrSavedCCodeFile);
            }
            else
            {
                // build a path out of the app startup directory and the default dir
                defaultDirectory = Path.Combine(Application.StartupPath, DEFAULT_CCODE_DIR);
            }

            // now sort out a filename
            if(currentCCodeFile!=null) defaultFileName = Path.GetFileNameWithoutExtension(currentCCodeFile);
            // did we get one
            if ((defaultFileName == null) || (defaultFileName.Length==0))
            {
                defaultFileName = DEFAULT_CCODE_FILENAME;
            }

            // build the dialog
            SaveFileDialog saveFileDialog1 = new SaveFileDialog();

            // NOTE: if we do not have the initial directory here just ignore
            // the operating system seems to remember it (on XP at least)
            if (defaultDirectory != null) saveFileDialog1.InitialDirectory = defaultDirectory;
            if (defaultFileName != null) saveFileDialog1.FileName = defaultFileName;
            saveFileDialog1.Filter = "CCode files (*." + DEFAULT_CCODE_EXT + ")|*." + DEFAULT_CCODE_EXT;
            saveFileDialog1.OverwritePrompt = false;
            saveFileDialog1.CreatePrompt = false;
            saveFileDialog1.FilterIndex = 1;
            saveFileDialog1.RestoreDirectory = true;

            if (saveFileDialog1.ShowDialog() != DialogResult.OK)
            {
                // user must have cancelled. Thats OK we just leave now
                ccodeFileName = null;
                return 10;
            }

            ccodeFileName = saveFileDialog1.FileName;
            return 0;
        }
        #endregion

        // ########################################################################
        // ##### ConnectionManagement - code which manages the TCP/IP connection
        // ########################################################################

        #region ConnectionManagement

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Makes or breaks the connection to the SRV1 - depending on whether
        /// we are currently connected or not
        /// </summary>
        /// <history>
        ///    25 Nov 08  Cynic - Originally written
        /// </history>
        private void ConnectToSRV1()
        {
            try
            {
                // do we want to connect?
                if (this.IsConnected == true)
                {
                    DiagnosticDialog("Already connected to the SRV1.\n\nPlease disconnect first");
                    return;
                }
                if ((textBoxIPAddress.Text == null) || (textBoxIPAddress.Text.Trim().Length == 0))
                {
                    DiagnosticDialog("The TCP address to connect to is empty.");
                    return;
                }
                if ((textBoxIPPort.Text == null) || (textBoxIPPort.Text.Trim().Length == 0))
                {
                    DiagnosticDialog("The IP Port to connect to is empty.");
                    return;
                }
                // always init this
                motorInitSent = false;
                lastNonImageCommandSent = null;
                // yes we do, setup the socket
                int alPort = System.Convert.ToInt16(textBoxIPPort.Text, 10);
                tcpClient = new TcpClient(textBoxIPAddress.Text, alPort);

                // Get a client stream for reading and writing.
                tcpBufferedStream = new TCPBufferedStream();
                tcpBufferedStream.NetStream = tcpClient.GetStream();
                tcpBufferedStream.NetStream.ReadTimeout = Timeout.Infinite;
                return;
            }
            catch (Exception ex)
            {
                DiagnosticDialog(ex);
            }
            finally
            {
                // set the display to reflect reality
                SyncConnectionStatusDisplay();
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Utility property to disconnect from the socket. We do this in multiple
        /// places so this provides common code
        /// </summary>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private void Disconnect()
        {
            // always reset this
            motorInitSent = false;
            lastNonImageCommandSent = null;
            // and this
            if(tcpBufferedStream!=null)
            {
                if (tcpBufferedStream.NetStream != null)
                {
                    tcpBufferedStream.NetStream.Close();
                }
                tcpBufferedStream.NetStream = null;
            }
            tcpBufferedStream=null;
            // we want to disconnect
            if (tcpClient != null)
            {
                tcpClient.Close();
            }
            tcpClient = null;
            SyncConnectionStatusDisplay();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Utility property to route all decisions about whether we are connected or
        /// not through a common point to avoid reproducing code
        /// </summary>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private bool IsConnected
        {
            get
            {
                if (tcpBufferedStream == null) return false;
                if (tcpBufferedStream.NetStream == null) return false;
                if (tcpClient == null) return false;
                return tcpClient.Connected;
            }
        }
        #endregion

        // ########################################################################
        // ##### Misc.FormManagement - code which manages the form
        // ########################################################################

        #region Misc.FormManagement

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Clears the error window
        /// </summary>
        /// <history>
        ///    03 Dec 08  Cynic - Originally written
        /// </history>
        private void ClearErrorWindow()
        {
            richTextBoxCCodeErrors.Text = "";
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handle the form closing operation by making sure we are disconnected
        /// </summary>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private void frmMain_FormClosing(object sender, FormClosingEventArgs e)
        {
            KillHeartbeat();
            if (this.IsConnected == true)
            {
                Disconnect();
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// The Connect/Disconnect operation is a tooggle. This synchronizes the 
        /// screen state (what it says) to the real connected state.
        /// </summary>
        /// <history>
        ///    19 Nov 08  Cynic - Originally written
        /// </history>
        private void SyncConnectionStatusDisplay()
        {
            if (this.IsConnected == true)
            {
                labelConnectionStatus.Text = "Connected";
                buttonToggleConnect.Text="Disconnect...";
            }
            else
            {
                labelConnectionStatus.Text = "Not Connected";
                buttonToggleConnect.Text="Connect...";
            }
        }
        #endregion

        // ########################################################################
        // ##### HeartbeatCode - code to handle the heartbeat thread
        // ########################################################################

        #region HeartbeatCode

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// The heartbeat is a continuously cycling thread which we use to 
        /// handle the automatic consumption of responses from the 
        /// SRV1
        /// </summary>
        /// <history>
        ///   24 Nov 08  Cynic - Originally written
        /// </history>
        private void LaunchHeartBeat()
        {
            // never start two of them
            if (heartBeatThread != null) return;
            ThreadStart heartBeatJob = new ThreadStart(HeartBeat);
            heartBeatThread = new Thread(heartBeatJob);
            heartBeatThread.Start();
        }
        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Kills off the heartbeat. Only called when the app is shutting down
        /// </summary>
        /// <history>
        ///   24 Nov 08  Cynic - Originally written
        /// </history>
        public void KillHeartbeat()
        {
            // kill off the heartbeat - it will trap this exception
            stopHeartBeat = true;
            if (heartBeatThread != null)
            {
                heartBeatThread.Abort();
                heartBeatThread = null;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// The heart beat handles repetitive tasks which need to happen periodically
        /// If you call anything which is going to interact with a form or form
        /// control you must set up a Invoke delegate mechanism otherwise you will
        /// "non thread safe" locking issues. See the comments below.
        /// </summary>
        /// <history>
        ///   24 Nov 08  Cynic - Originally written
        /// </history>
        private void HeartBeat()
        {
            try
            {
                // this is where the heart beat does its stuff we do not
                // stop till we get a Thread.Abort
                while (true)
                {
                    // this will wake up every so often to perform its
                    // required processing
                    // the the processing of the rule queue
                    try
                    {
                        Thread.Sleep(HEARTBEAT_DELAYTIME);

                        /* Note: we do not have to lock access to the tcpStream
                         * because we use Invoke to run everything on the main 
                         * form thread. The heartbeat thread will block 
                         * if the main form is busy - forming an implicit 
                         * lock. If you wish to convince yourself of this, enable
                         * the MessageBeep() line below and then copy the 
                         * two sendstring and the delay statement to the Test1 button
                         * click handler. You will hear the beeping stop while the
                         * main form is performing the 5 second sleep */
                        // MessageBeep(0);
                            // diagnostic code for test1 button handler
                            //SendSRV1CommandAsStr(MOTOR_BACK_ASSTR);
                            //Thread.Sleep(5000);
                            //SendSRV1CommandAsStr(MOTOR_STOP_ASSTR);

                        // should we exit
                        if (stopHeartBeat == true) break;

                        // processes the SRV1 responses
                        // notice we use Invoke here rather than BeginInvoke
                        // we do want to block on this call. kicking this off
                        // asynchronously could back up a lot of messages 
                        this.Invoke(MonitorSRV1Response);

                        // should we exit
                        if (stopHeartBeat == true) break;

                        // do we want to request image data from the SRV1
                        if (wantCameraOutput == true)
                        {
                            // yes, we do, use invoke to send the command to
                            // the SRV1 on the appropriate form thread
                            this.Invoke(SendSRV1Command, new Object[] { IMAGE_FETCH_ASSTR });
                        }
                    }
                    catch (ThreadAbortException)
                    {
                        // this just means the heartbeat stopped
                        break;
                    }
                    catch (Exception)
                    {
                    }
                } // bottom of while(true)
            }
            catch (ThreadAbortException)
            {
            }
            finally
            {
                heartBeatThread = null;
            }
        }

        #endregion

        // ########################################################################
        // ##### DiagnosticCode - code to handle the posting of diagnostics
        // ########################################################################

        #region DiagnosticCode

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A utility proc to accept an Exception and pump out a display message
        /// </summary>
        /// <history>
        ///    19 Nov 08  Cynic - Originally written
        /// </history>
        private void DiagnosticDialog(Exception ex)
        {
            // we must be on the form thread!!!
            if (InvokeRequired == true)
            {
                // only call from the form thread 
                return;
            }

            if (ex == null) return;
            StringBuilder sb = new StringBuilder();
            sb.Append("Exception happened\n\n");
            if (ex.Message != null) sb.Append(ex.Message);
            sb.Append("\n\n");
            if (ex.StackTrace != null) sb.Append(ex.StackTrace);
            // log it
            LogMessage(sb.ToString());
            // display it
            MessageBox.Show(sb.ToString());
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A utility proc to accept a string and pump out a display message
        /// </summary>
        /// <history>
        ///    19 Nov 08  Cynic - Originally written
        /// </history>
        private void DiagnosticDialog(string text)
        {
            // we must be on the form thread!!!
            if (InvokeRequired == true)
            {
                // only call from the form thread 
                return;
            }

            if (text == null) return;
            // log it
            LogMessage(text);
            // display it
            MessageBox.Show(text);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A property which always tells us the max number of lines allowed in the
        /// logfile window
        /// </summary>
        /// <history>
        ///    21 Nov 08  Cynic - Originally written
        /// </history>
       private int MaxLogFileLines
        {
            get
            {
                // just this for now
                return DEFAULT_MAX_LOGFILE_LINES;
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A utility proc to accept a string and pump out a message to the log 
        /// window
        /// </summary>
        /// <history>
        ///    21 Nov 08  Cynic - Originally written
        /// </history>
        private void LogMessage(string logMessage)
        {

            // we must be on the form thread!!!
            if (InvokeRequired == true)
            {
                // only call from the form thread 
                return;
            }

            // sanity checks
            if (logMessage == null) return;
            if (logMessage.Length==0) return;
            // remove any CR chars
            string text=logMessage.Replace("\r","");
            if (text == null) return;
            if (text.Length==0) return;

            // add on our new message
            if (text.EndsWith("\n") == false)
            {
                // add it with a trailing return
                richTextBoxLogfile.Text+= text + "\n";
            }
            else
            {
                // just add it with a trailing return
                richTextBoxLogfile.Text += text;
            }

            // trim the size back
            SetMaxLinesInLogWindow(MaxLogFileLines);

        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Removes enough lines from the logfile so we do not exceed the max
        /// </summary>
        /// <history>
        ///    21 Nov 08  Cynic - Originally written
        /// </history>
        private void SetMaxLinesInLogWindow(int maxLines)
        {
            int lineCount = 0;
            int removeCount = 0;
            // sanity check
            if (maxLines <= 0) return;
            // how many have we got?
            lineCount=this.richTextBoxLogfile.Lines.GetLength(0);
            // got less than or equal - that is ok
            if (lineCount <= maxLines) return;
            // we remove this many
            removeCount = lineCount - maxLines;
            for (int i = 0; i < removeCount; i++)
            {
                RemoveTopMostLogfileLine();
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Removes the topmost line from the logfile.
        /// </summary>
        /// <history>
        ///    21 Nov 08  Cynic - Originally written
        /// </history>
        public void RemoveTopMostLogfileLine()
        {
            // Delete the topmost line in the most sensible way
            // this will minimize jitter and flashing
            richTextBoxLogfile.SelectionStart = 0;
            richTextBoxLogfile.SelectionLength = richTextBoxLogfile.Text.IndexOf("\n", 0) + 1;
            richTextBoxLogfile.SelectedText = "";
        }
        #endregion

        // ########################################################################
        // ##### UtilityCode - code to provide useful conversions & translations etc
        // ########################################################################

        #region UtilityCode

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A primitive detector of binary returning data 
        /// </summary>
        /// <param name="testStr">The string to test</param>
        /// <returns>the number of non printable chars in a string</returns>
        /// <history>
        ///    26 Nov 08  Cynic - Originally written
        /// </history>
        private int CountNonPrintableCharsInString(string testStr)
        {
            int count = 0;
            if (testStr == null) return 0;
            foreach (char tmpChar in testStr)
            {
                if ((tmpChar < ' ') || (tmpChar > '~'))
                {
                    // count the unprintable chars
                    count++;
                }
            }
            return count;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A function to four input bytes to an int. Each successive byte is
        /// scaled by a power of 256 (the returning image string encodes
        /// information in this way).
        /// </summary>
        /// <param name="byte0">The 256^0 byte value</param>
        /// <param name="byte1">The 256^1 byte value</param>
        /// <param name="byte2">The 256^2 byte value</param>
        /// <param name="byte3">The 256^3 byte value</param>
        /// <returns>the bytes converted to an int</returns>
        /// <history>
        ///    26 Nov 08  Cynic - Originally written
        /// </history>
        private int ConvertFourBytesToScaledInt(byte byte0, byte byte1, byte byte2, byte byte3)
        {
            int b1 = Convert.ToInt32(byte0) * (1);
            int b256 = Convert.ToInt32(byte1) * (256);
            int b256x256 = Convert.ToInt32(byte2) * (256 * 256);
            int b256x256x256 = Convert.ToInt32(byte3) * (256 * 256 * 256);
            return b1 + b256 + b256x256 + b256x256x256;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A function to convert hex strings into a binary array.
        /// </summary>
        /// <param name="spacedString">A hex string with space or comma delimiters
        /// for example "0x03 0x34,0xfd"</param>
        /// <returns>an array of bytes of the converted string</returns>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private byte [] ConvertSpacedHexStringToByteArray(string spacedString)
        {
            string workingStr = null;
            byte [] byteArr=null;
            string[] splitArr=null;
            // these are the acceptable delimiters of the hex string
            char[] splitter = {' ',','};

            // sanity checks - should never happen
            if (spacedString == null)
            {
                // should never happen
                DiagnosticDialog("SendMotorCommand Err1: Bad Input Param");
                return null;
            }
            // trim it off and re-check
            workingStr = spacedString.Trim();
            if (workingStr.Length == 0)
            {
                // should never happen
                DiagnosticDialog("SendMotorCommand Err2: Bad Input Param");
                return null;
            }

            // bust it on the delimiters
            splitArr = workingStr.Split(splitter);
            // did we find anything?
            if(splitArr==null) return null;
            if(splitArr.Length==0) return null;

            // by the time we get here we should have an array of string each 
            // containingsingle strings representing hex values. Now create the
            // output byte array
            byteArr=new byte[splitArr.Length];
            // now populate the byteArr with converted hex strings
            for(int i=0;i<splitArr.Length; i++)
            {
                try
                {
                    byteArr[i] = byte.Parse(splitArr[i], System.Globalization.NumberStyles.HexNumber);
                }
                catch (Exception ex)
                {
                    // tell the user
                    DiagnosticDialog(ex);
                    return null;
                }
            }
            // we are all done - hand it back
            return byteArr;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Make a Beep sound - just essentially the Win32 MessageBeep()
        /// </summary>
        /// <history>
        ///    03 Dec 08  Cynic - Originally written
        /// </history>
        [DllImport("user32")]
        public static extern int MessageBeep(int wType);
        #endregion

        // ########################################################################
        // ##### ButtonCode - button handlers
        // ########################################################################

        #region ButtonCode

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// For testing and debug
        /// </summary>
        /// <history>
        ///    21 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonTest1_Click(object sender, EventArgs e)
        {
            DiagnosticDialog("This button does not do anything (until you make it do something).");
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// For testing and debug
        /// </summary>
        /// <history>
        ///    27 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonTest2_Click(object sender, EventArgs e)
        {
             DiagnosticDialog("This button does not do anything (until you make it do something).");
       }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a key press on the SendAsciiCode button
        /// </summary>
        /// <history>
        ///    19 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonSendAsciiCode_Click(object sender, EventArgs e)
        {
            int retInt;

            if ((textBoxAsciiCode.Text == null) || (textBoxAsciiCode.Text.Length == 0))
            {
                DiagnosticDialog("There is no text to send");
                return;
            }
            if (this.IsConnected==false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            try
            {
                // send it now
                retInt=SendSRV1CommandAsStr(textBoxAsciiCode.Text);
                if (retInt!=0)
                {
                    DiagnosticDialog("Error "+retInt.ToString()+" occurred sending the ascii codes.");
                    return;
                }
            }
            catch (Exception ex)
            {
                DiagnosticDialog(ex);
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a key press on the SendHexBytes button
        /// </summary>
        /// <history>
        ///    01 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonSendHexBytes_Click(object sender, EventArgs e)
        {
            int retInt;

            if ((textBoxHexBytes.Text == null) || (textBoxHexBytes.Text.Length == 0))
            {
                DiagnosticDialog("There are no hex bytes to send");
                return;
            }
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            try
            {
                // send it now
                retInt = SendSRV1CommandAsHexStr(textBoxHexBytes.Text);
                if (retInt != 0)
                {
                    DiagnosticDialog("Error " + retInt.ToString() + " occurred sending the hex bytes.");
                    return;
                }
            }
            catch (Exception ex)
            {
                DiagnosticDialog(ex);
            }
        }


        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Handles a key press on the Connect button
        /// </summary>
        /// <history>
        ///    19 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonToggleConnect_Click(object sender, EventArgs e)
        {
            // do we want to connect?
            if (this.IsConnected == false)
            {
                ConnectToSRV1();
            }
            else
            {
                Disconnect();
            }
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A proc to handle keypresses on the close button
        /// </summary>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonClose_Click(object sender, EventArgs e)
        {
            this.Disconnect();
            this.Close();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// A proc to handle keypresses on this button
        /// </summary>
        /// <history>
        ///    20 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonForward_Click(object sender, EventArgs e)
        {
            SendSimpleMotorCommand(MOTOR_FORWARD_ASSTR);
        }

        private void buttonStop_Click(object sender, EventArgs e)
        {
            SendSimpleMotorCommand(MOTOR_STOP_ASSTR);
        }

        private void buttonBack_Click(object sender, EventArgs e)
        {
            SendSimpleMotorCommand(MOTOR_BACK_ASSTR);
        }

        private void buttonDriftLeft_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_DRIFTLEFT_ASHEXSTR);
        }

        private void buttonDriftRight_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_DRIFTRIGHT_ASHEXSTR);
        }

        private void buttonTurnLeft_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_TURNLEFT_ASHEXSTR);
        }

        private void buttonTurnRight_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_TURNRIGHT_ASHEXSTR);
        }

        private void buttonBackRight_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_BACKRIGHT_ASHEXSTR);
        }

        private void buttonBackLeft_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_BACKLEFT_ASHEXSTR);
        }

        private void buttonTurnCW_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_TURNCW_ASHEXSTR);
        }

        private void buttonTurnCCW_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsHexStr(MOTOR_TURNCCW_ASHEXSTR);
        }

        private void buttonLaserOn_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsStr(LASER_ON_ASSTR);
        }

        private void buttonLaserOff_Click(object sender, EventArgs e)
        {
            SendSRV1CommandAsStr(LASER_OFF_ASSTR);
        }

        private void buttonCamera1280x1024_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            SendSRV1CommandAsStr(IMAGE_SIZE_1280x1024_ASSTR);
            wantCameraOutput = true;
        }

        private void buttonCamera640x480_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            SendSRV1CommandAsStr(IMAGE_SIZE_640x480_ASSTR);
            wantCameraOutput = true;
        }

        private void buttonCamera320x240_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            SendSRV1CommandAsStr(IMAGE_SIZE_320x240_ASSTR);
            wantCameraOutput = true;
        }

        private void buttonCamera160x120_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            SendSRV1CommandAsStr(IMAGE_SIZE_160x120_ASSTR);
            wantCameraOutput = true;
        }

        private void buttonCameraNoImage_Click(object sender, EventArgs e)
        {
            // this is all we need to do to turn it off. The heartbeat
            // will stop requesting image updates if this is off
            wantCameraOutput = false;
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Sends the contents of the CCode rich text box to the SRV1. We just
        /// use the simple SRV1 line editor for this. Seems a lot easier than
        /// XModem transfers
        /// </summary>
        /// <history>
        ///    28 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonSendCCode_Click(object sender, EventArgs e)
        {
            // do we have any code?
            if ((richTextBoxCCode.Text == null) || (richTextBoxCCode.Text.Length == 0))
            {
                DiagnosticDialog("There is no C code to send to the SRV1 in the panel.");
                return; 
            }
            // also need to check this
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            // send the string now
            SendCodeByLineEditor(richTextBoxCCode.Text);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Gets the contents of the SRV1 flash buffer and displays it in
        /// a modeless form
        /// </summary>
        /// <history>
        ///    28 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonViewFlashBuffer_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }

            string codeStr = GetSRV1FlashBufferContents();
            // tidy up a bit and put in some CRLF's for the LF's
            if (codeStr != null)
            {
                codeStr = codeStr.Replace("zdump: \n", "");
                codeStr = codeStr.Replace("\n", "\r\n");
            }
            frmSRV1FlashBufferCode frmCode = new frmSRV1FlashBufferCode(codeStr);
            frmCode.Show();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Runs the code in the SRV1 flash buffer
        /// </summary>
        /// <history>
        ///    27 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonRunFlashBuffer_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            // this does it all
            RunSRV1Buffer();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Clears out the SRV1 flash buffer
        /// </summary>
        /// <history>
        ///    27 Nov 08  Cynic - Originally written
        /// </history>
        private void buttonClearFlashBuffer_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            // just send a zc command to clean out whatever is in there
            SendSRV1CommandAsStr(CLEAR_FLASHBUFFER_ASSTR);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Clears the log
        /// </summary>
        /// <history>
        ///    01 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonClearLog_Click(object sender, EventArgs e)
        {
            richTextBoxLogfile.Text = "";
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Clears error window
        /// </summary>
        /// <history>
        ///    03 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonClearErrorWindow_Click(object sender, EventArgs e)
        {
            ClearErrorWindow();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Reads 65k from the flash sector to the flash buffer
        /// </summary>
        /// <history>
        ///    03 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonReadFromFlashSector_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            // just send a zr command to read whatever is in 
            // the flash sector into the flash buffer where it can
            // be run with a zc command
            SendSRV1CommandAsStr(READFLASHSECTORINTOFLASHBUFFER_ASSTR);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Writes 65k from the flash buffer to the flash sector
        /// </summary>
        /// <history>
        ///    03 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonWriteToFlashSector_Click(object sender, EventArgs e)
        {
            if (this.IsConnected == false)
            {
                DiagnosticDialog("There is no connection to the SRV1");
                return;
            }
            // just send a zw command to write whatever is in 
            // the flash buffer into the flash sector so it is
            // remembered over power offs 
            SendSRV1CommandAsStr(WRITEFLASHBUFFERINTOFLASHSECTOR_ASSTR);
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Reads the contents of the ccode buffer from a file
        /// </summary>
        /// <history>
        ///    05 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonReadCCodeFromFile_Click(object sender, EventArgs e)
        {
            ReadFile();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Writes the contents of the ccode buffer to a file only prompts if
        /// there is no filename
        /// </summary>
        /// <history>
        ///    04 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonSaveCCodeToFile_Click(object sender, EventArgs e)
        {
            SaveFile();
        }

        /// +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=
        /// <summary>
        /// Writes the contents of the ccode buffer to a file, always prompts
        /// </summary>
        /// <history>
        ///    03 Dec 08  Cynic - Originally written
        /// </history>
        private void buttonSaveAsCCodeToFile_Click(object sender, EventArgs e)
        {
            SaveFileAs();
        }

        #endregion

    }
}